//
//  bt_spec_gatt.h
//
//  Bluetooth Protocol Stack - GATT(Generic Attribute Profile) Specification definition
//  Copyright (C) 2013-2017 Toyohiko Togashi tog001@nifty.com
//
//
//  This program is free software; you can redistribute it and/or modify it under the terms of the
//  GNU General Public License as published by the Free Software Foundation; either version 3
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
//  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with this program.
//  If not, see <http://www.gnu.org/licenses/>
//
//
//  Reference:
//      Bluetooth SIG (www.bluetooth.com)
//          BLUETOOTH SPECIFICATION Version 4.0
//          [Vol 3] Core System Package [Host volume]
//          Part G: GENERIC ATTRIBUTE PROFILE (GATT)
//          https://www.bluetooth.com/specifications/gatt
//
//  Update history
//  ---------- ----- -------------------------------------------
//  2013.01.21 v0.0  First cording
//  2013.04.21 v0.1  Commit
//  2017.02.06 v0.3g Appended: GATT_Characteristic_Descriptors-
//  2017.02.06 v0.3g Appended: GATT_Characteristic_Appearance_Category-
//  2017.02.11 v0.3h Appended: GATT_services,GATT_attribute,GATT_descriptor,GATT_characteristic
//
#ifndef BT_SPEC_GATT_H_
#define BT_SPEC_GATT_H_

#include <stdint.h>
//#include "bt_spec_att.h"      // DEL 2017.02.11

#define GATT_Services_Generic_Access_Profile        0x1800
#define GATT_Services_Generic_Attribute_Profile     0x1801
#define GATT_services_generic_access                0x1800  // https://www.bluetooth.com/specifications/gatt/services
#define GATT_services_generic_attribute             0x1801  //
#define GATT_services_immediate_alert               0x1802  //
#define GATT_services_link_loss                     0x1803  //
#define GATT_services_tx_power                      0x1804  //
#define GATT_services_current_time                  0x1805  //
#define GATT_services_reference_time_update         0x1806  //
#define GATT_services_next_dst_change               0x1807  //
#define GATT_services_glucose                       0x1808  //
#define GATT_services_health_thermometer            0x1809  //
#define GATT_services_device_information            0x180A  //
#define GATT_services_heart_rate                    0x180D  //
#define GATT_services_phone_alert_status            0x180E  //
#define GATT_services_battery_service               0x180F  //
#define GATT_services_blood_pressure                0x1810  //
#define GATT_services_alert_notification            0x1811  //
#define GATT_services_human_interface_device        0x1812  //
#define GATT_services_scan_parameters               0x1813  //
#define GATT_services_running_speed_and_cadence     0x1814  //
#define GATT_services_automation_io                 0x1815  //
#define GATT_services_cycling_speed_and_cadence     0x1816  //
#define GATT_services_cycling_power                 0x1818  //
#define GATT_services_location_and_navigation       0x1819  //
#define GATT_services_environmental_sensing         0x181A  //
#define GATT_services_body_composition              0x181B  //
#define GATT_services_user_data                     0x181C  //
#define GATT_services_weight_scale                  0x181D  //
#define GATT_services_bond_management               0x181E  //
#define GATT_services_continuous_glucose_monitoring 0x181F  //
#define GATT_services_internet_protocol_support     0x1820  //
#define GATT_services_indoor_positioning            0x1821  //
#define GATT_services_pulse_oximeter                0x1822  //
#define GATT_services_http_proxy                    0x1823  //
#define GATT_services_transport_discovery           0x1824  //
#define GATT_services_object_transfer               0x1825  //

#define GATT_Attribute_Types_Primary_Service                0x2800  // Value={UUID16/128bits}, R/O, NoAuth
#define GATT_Attribute_Types_Secondary_Service              0x2801  // Value={UUID16/128bits}, R/O, NoAuth
#define GATT_Attribute_Types_Include                        0x2802  // Value={Included Service Attribute Handle, End Group Handle, Service UUID}, R/O, NoAuth
#define GATT_Attribute_Types_Characteristic                 0x2803  // Value={Characteristic Properties, Characteristic Value Attribute Handle, Characteristic UUID}, R/O, NoAuth
// https://www.bluetooth.com/specifications/gatt/declarations
#define GATT_attribute_gatt_primary_service_declaration     0x2800  //
#define GATT_attribute_gatt_secondary_service_declaration   0x2801  //
#define GATT_attribute_gatt_include_declaration             0x2802  //
#define GATT_attribute_gatt_characteristic_declaration      0x2803  //

#define GATT_Characteristic_Descriptors_Characteristic_Extended_Properties  0x2900  // Value=2bytes, R/O, NoAuth
#define GATT_Characteristic_Descriptors_Characteristic_User_Description     0x2901  // Value="string"
#define GATT_Characteristic_Descriptors_Client_Characteristic_Configuration 0x2902  // Value=2bytes(default="shall be 0x0000"), R/W
#define GATT_Characteristic_Descriptors_Server_Characteristic_Configuration 0x2903  // Value=2bytes(default="shall be 0x0000"), R/W
#define GATT_Characteristic_Descriptors_Characteristic_Format               0x2904  // Value={Format,Exponent,Unit,NameSpace,Description}, R/O, NoAuth
// https://www.bluetooth.com/specifications/gatt/descriptors
#define GATT_descriptor_gatt_characteristic_extended_properties             0x2900  //
#define GATT_descriptor_gatt_characteristic_user_description                0x2901  //
#define GATT_descriptor_gatt_client_characteristic_configuration            0x2902  //
#define GATT_descriptor_gatt_server_characteristic_configuration            0x2903  //
#define GATT_descriptor_gatt_characteristic_presentation_format             0x2904  //
#define GATT_descriptor_gatt_characteristic_aggregate_format                0x2905  //
#define GATT_descriptor_valid_range                                         0x2906  //
#define GATT_descriptor_external_report_reference                           0x2907  //
#define GATT_descriptor_report_reference                                    0x2908  //
#define GATT_descriptor_number_of_digitals                                  0x2909  //  uint8
#define GATT_descriptor_value_trigger_setting                               0x290A  //
#define GATT_descriptor_es_configuration                                    0x290B  //
#define GATT_descriptor_es_measurement                                      0x290C  //
#define GATT_descriptor_es_trigger_setting                                  0x290D  //
#define GATT_descriptor_time_trigger_setting                                0x290E  //

#define GATT_Characteristic_Types_Device_Name                               0x2a00
#define GATT_Characteristic_Types_Appearance                                0x2a01
#define GATT_Characteristic_Types_Peripheral_Privacy_Flag                   0x2a02
#define GATT_Characteristic_Types_Reconnection_Address                      0x2a03
#define GATT_Characteristic_Types_Peripheral_Preferred_Connection_Parameters 0x2a04
#define GATT_Characteristic_Types_Service_Changed                           0x2a05
// https://www.bluetooth.com/specifications/gatt/characteristics
#define GATT_characteristic_gap_device_name                                 0x2A00  //
#define GATT_characteristic_gap_appearance                                  0x2A01  //
#define GATT_characteristic_gap_peripheral_privacy_flag                     0x2A02  //
#define GATT_characteristic_gap_reconnection_address                        0x2A03  //
#define GATT_characteristic_gap_peripheral_preferred_connection_parameters  0x2A04  //
#define GATT_characteristic_gatt_service_changed                            0x2A05  //
#define GATT_characteristic_alert_level                                     0x2A06  //
#define GATT_characteristic_tx_power_level                                  0x2A07  //
#define GATT_characteristic_date_time                                       0x2A08  //
#define GATT_characteristic_day_of_week                                     0x2A09  //
#define GATT_characteristic_day_date_time                                   0x2A0A  //
#define GATT_characteristic_exact_time_256                                  0x2A0C  //
#define GATT_characteristic_dst_offset                                      0x2A0D  //
#define GATT_characteristic_time_zone                                       0x2A0E  //
#define GATT_characteristic_local_time_information                          0x2A0F  //
#define GATT_characteristic_time_with_dst                                   0x2A11  //
#define GATT_characteristic_time_accuracy                                   0x2A12  //
#define GATT_characteristic_time_source                                     0x2A13  //
#define GATT_characteristic_reference_time_information                      0x2A14  //
#define GATT_characteristic_time_update_control_point                       0x2A16  //
#define GATT_characteristic_time_update_state                               0x2A17  //
#define GATT_characteristic_glucose_measurement                             0x2A18  //
#define GATT_characteristic_battery_level                                   0x2A19  //
#define GATT_characteristic_temperature_measurement                         0x2A1C  //
#define GATT_characteristic_temperature_type                                0x2A1D  //
#define GATT_characteristic_intermediate_temperature                        0x2A1E  //
#define GATT_characteristic_measurement_interval                            0x2A21  //
#define GATT_characteristic_boot_keyboard_input_report                      0x2A22  //
#define GATT_characteristic_system_id                                       0x2A23  //
#define GATT_characteristic_model_number_string                             0x2A24  //
#define GATT_characteristic_serial_number_string                            0x2A25  //
#define GATT_characteristic_firmware_revision_string                        0x2A26  //
#define GATT_characteristic_hardware_revision_string                        0x2A27  //
#define GATT_characteristic_software_revision_string                        0x2A28  //
#define GATT_characteristic_manufacturer_name_string                        0x2A29  //
#define GATT_characteristic_ieee_11073_20601_regulatory_certification_data_list 0x2A2A //
#define GATT_characteristic_current_time                                    0x2A2B  //
#define GATT_characteristic_magnetic_declination                            0x2A2C  //
#define GATT_characteristic_scan_refresh                                    0x2A31  //
#define GATT_characteristic_boot_keyboard_output_report                     0x2A32  //
#define GATT_characteristic_boot_mouse_input_report                         0x2A33  //
#define GATT_characteristic_glucose_measurement_context                     0x2A34  //
#define GATT_characteristic_blood_pressure_measurement                      0x2A35  //
#define GATT_characteristic_intermediate_cuff_pressure                      0x2A36  //
#define GATT_characteristic_heart_rate_measurement                          0x2A37  //
#define GATT_characteristic_body_sensor_location                            0x2A38  //
#define GATT_characteristic_heart_rate_control_point                        0x2A39  //
#define GATT_characteristic_alert_status                                    0x2A3F  //
#define GATT_characteristic_ringer_control_point                            0x2A40  //
#define GATT_characteristic_ringer_setting                                  0x2A41  //
#define GATT_characteristic_alert_category_id_bit_mask                      0x2A42  //
#define GATT_characteristic_alert_category_id                               0x2A43  //
#define GATT_characteristic_alert_notification_control_point                0x2A44  //
#define GATT_characteristic_unread_alert_status                             0x2A45  //
#define GATT_characteristic_new_alert                                       0x2A46  //
#define GATT_characteristic_supported_new_alert_category                    0x2A47  //
#define GATT_characteristic_supported_unread_alert_category                 0x2A48  //
#define GATT_characteristic_blood_pressure_feature                          0x2A49  //
#define GATT_characteristic_hid_information                                 0x2A4A  //
#define GATT_characteristic_report_map                                      0x2A4B  //
#define GATT_characteristic_hid_control_point                               0x2A4C  //
#define GATT_characteristic_report                                          0x2A4D  //
#define GATT_characteristic_protocol_mode                                   0x2A4E  //
#define GATT_characteristic_scan_interval_window                            0x2A4F  //
#define GATT_characteristic_pnp_id                                          0x2A50  //
#define GATT_characteristic_glucose_feature                                 0x2A51  //
#define GATT_characteristic_record_access_control_point                     0x2A52  //
#define GATT_characteristic_rsc_measurement                                 0x2A53  //
#define GATT_characteristic_rsc_feature                                     0x2A54  //
#define GATT_characteristic_sc_control_point                                0x2A55  //
#define GATT_characteristic_digital                                         0x2A56  // 2bit * n (array)
#define GATT_characteristic_analog                                          0x2A58  // uint16
#define GATT_characteristic_aggregate                                       0x2A5A  //
#define GATT_characteristic_csc_measurement                                 0x2A5B  //
#define GATT_characteristic_csc_feature                                     0x2A5C  //
#define GATT_characteristic_sensor_location                                 0x2A5D  //
#define GATT_characteristic_plx_spot_check_measurement                      0x2A5E  //
#define GATT_characteristic_plx_continuous_measurement                      0x2A5F  //
#define GATT_characteristic_plx_features                                    0x2A60  //
#define GATT_characteristic_cycling_power_measurement                       0x2A63  //
#define GATT_characteristic_cycling_power_vector                            0x2A64  //
#define GATT_characteristic_cycling_power_feature                           0x2A65  //
#define GATT_characteristic_cycling_power_control_point                     0x2A66  //
#define GATT_characteristic_location_and_speed                              0x2A67  //
#define GATT_characteristic_navigation                                      0x2A68  //
#define GATT_characteristic_position_quality                                0x2A69  //
#define GATT_characteristic_ln_feature                                      0x2A6A  //
#define GATT_characteristic_ln_control_point                                0x2A6B  //
#define GATT_characteristic_elevation                                       0x2A6C  //
#define GATT_characteristic_pressure                                        0x2A6D  //
#define GATT_characteristic_temperature                                     0x2A6E  //
#define GATT_characteristic_humidity                                        0x2A6F  //
#define GATT_characteristic_true_wind_speed                                 0x2A70  //
#define GATT_characteristic_true_wind_direction                             0x2A71  //
#define GATT_characteristic_apparent_wind_speed                             0x2A72  //
#define GATT_characteristic_apparent_wind_direction                         0x2A73  //
#define GATT_characteristic_gust_factor                                     0x2A74  //
#define GATT_characteristic_pollen_concentration                            0x2A75  //
#define GATT_characteristic_uv_index                                        0x2A76  //
#define GATT_characteristic_irradiance                                      0x2A77  //
#define GATT_characteristic_rainfall                                        0x2A78  //
#define GATT_characteristic_wind_chill                                      0x2A79  //
#define GATT_characteristic_heat_index                                      0x2A7A  //
#define GATT_characteristic_dew_point                                       0x2A7B  //
#define GATT_characteristic_descriptor_value_changed                        0x2A7D  //
#define GATT_characteristic_aerobic_heart_rate_lower_limit                  0x2A7E  //
#define GATT_characteristic_aerobic_threshold                               0x2A7F  //
#define GATT_characteristic_age                                             0x2A80  //
#define GATT_characteristic_anaerobic_heart_rate_lower_limit                0x2A81  //
#define GATT_characteristic_anaerobic_heart_rate_upper_limit                0x2A82  //
#define GATT_characteristic_anaerobic_threshold                             0x2A83  //
#define GATT_characteristic_aerobic_heart_rate_upper_limit                  0x2A84  //
#define GATT_characteristic_date_of_birth                                   0x2A85  //
#define GATT_characteristic_date_of_threshold_assessment                    0x2A86  //
#define GATT_characteristic_email_address                                   0x2A87  //
#define GATT_characteristic_fat_burn_heart_rate_lower_limit                 0x2A88  //
#define GATT_characteristic_fat_burn_heart_rate_upper_limit                 0x2A89  //
#define GATT_characteristic_first_name                                      0x2A8A  //
#define GATT_characteristic_five_zone_heart_rate_limits                     0x2A8B  //
#define GATT_characteristic_gender                                          0x2A8C  //
#define GATT_characteristic_heart_rate_max                                  0x2A8D  //
#define GATT_characteristic_height                                          0x2A8E  //
#define GATT_characteristic_hip_circumference                               0x2A8F  //
#define GATT_characteristic_last_name                                       0x2A90  //
#define GATT_characteristic_maximum_recommended_heart_rate                  0x2A91  //
#define GATT_characteristic_resting_heart_rate                              0x2A92  //
#define GATT_characteristic_sport_type_for_aerobic_and_anaerobic_thresholds 0x2A93  //
#define GATT_characteristic_three_zone_heart_rate_limits                    0x2A94  //
#define GATT_characteristic_two_zone_heart_rate_limit                       0x2A95  //
#define GATT_characteristic_vo2_max                                         0x2A96  //
#define GATT_characteristic_waist_circumference                             0x2A97  //
#define GATT_characteristic_weight                                          0x2A98  //
#define GATT_characteristic_database_change_increment                       0x2A99  //
#define GATT_characteristic_user_index                                      0x2A9A  //
#define GATT_characteristic_body_composition_feature                        0x2A9B  //
#define GATT_characteristic_body_composition_measurement                    0x2A9C  //
#define GATT_characteristic_weight_measurement                              0x2A9D  //
#define GATT_characteristic_weight_scale_feature                            0x2A9E  //
#define GATT_characteristic_user_control_point                              0x2A9F  //
#define GATT_characteristic_magnetic_flux_density_2D                        0x2AA0  //
#define GATT_characteristic_magnetic_flux_density_3D                        0x2AA1  //
#define GATT_characteristic_language                                        0x2AA2  //
#define GATT_characteristic_barometric_pressure_trend                       0x2AA3  //
#define GATT_characteristic_bond_management_control_point                   0x2AA4  //
#define GATT_characteristic_bond_management_feature                         0x2AA5  //
#define GATT_characteristic_gap_central_address_resolution_support          0x2AA6  //
#define GATT_characteristic_cgm_measurement                                 0x2AA7  //
#define GATT_characteristic_cgm_feature                                     0x2AA8  //
#define GATT_characteristic_cgm_status                                      0x2AA9  //
#define GATT_characteristic_cgm_session_start_time                          0x2AAA  //
#define GATT_characteristic_cgm_session_run_time                            0x2AAB  //
#define GATT_characteristic_cgm_specific_ops_control_point                  0x2AAC  //
#define GATT_characteristic_indoor_positioning_configuration                0x2AAD  //
#define GATT_characteristic_latitude                                        0x2AAE  //
#define GATT_characteristic_longitude                                       0x2AAF  //
#define GATT_characteristic_local_north_coordinate                          0x2AB0  //
#define GATT_characteristic_local_east_coordinate                           0x2AB1  //
#define GATT_characteristic_floor_number                                    0x2AB2  //
#define GATT_characteristic_altitude                                        0x2AB3  //
#define GATT_characteristic_uncertainty                                     0x2AB4  //
#define GATT_characteristic_location_name                                   0x2AB5  //
#define GATT_characteristic_uri                                             0x2AB6  //
#define GATT_characteristic_http_headers                                    0x2AB7  //
#define GATT_characteristic_http_status_code                                0x2AB8  //
#define GATT_characteristic_http_entity_body                                0x2AB9  //
#define GATT_characteristic_http_control_point                              0x2ABA  //
#define GATT_characteristic_https_security                                  0x2ABB  //
#define GATT_characteristic_tds_control_point                               0x2ABC  //
#define GATT_characteristic_ots_feature                                     0x2ABD  //
#define GATT_characteristic_object_name                                     0x2ABE  //
#define GATT_characteristic_object_type                                     0x2ABF  //
#define GATT_characteristic_object_size                                     0x2AC0  //
#define GATT_characteristic_object_first_created                            0x2AC1  //
#define GATT_characteristic_object_last_modified                            0x2AC2  //
#define GATT_characteristic_object_id                                       0x2AC3  //
#define GATT_characteristic_object_properties                               0x2AC4  //
#define GATT_characteristic_object_action_control_point                     0x2AC5  //
#define GATT_characteristic_object_list_control_point                       0x2AC6  //
#define GATT_characteristic_object_list_filter                              0x2AC7  //
#define GATT_characteristic_object_changed                                  0x2AC8  //
#define GATT_characteristic_resolvable_private_address_only                 0x2AC9  //

// GATT_Attribute_Types_Characteristic's value#1
#define GATT_Characteristic_Properties_Broadcast                        0x01
#define GATT_Characteristic_Properties_Read                             0x02
#define GATT_Characteristic_Properties_Write_Without_Response           0x04
#define GATT_Characteristic_Properties_Write                            0x08
#define GATT_Characteristic_Properties_Notify                           0x10
#define GATT_Characteristic_Properties_Indicate                         0x20
#define GATT_Characteristic_Properties_Authenticated_Signed_Writes      0x40
#define GATT_Characteristic_Properties_Extended_Properties              0x80

// GATT_Characteristic_Descriptors_Characteristic_Extended_Properties's value
#define GATT_Characteristic_Extended_Properties_Reliable_Write          0x0001
#define GATT_Characteristic_Extended_Properties_Writable_Auxiliaries    0x0002

// GATT_Characteristic_Descriptors_Client_Characteristic_Configuration's value
#define GATT_Client_Characteristic_Configuration_Notification           0x0001
#define GATT_Client_Characteristic_Configuration_Indication             0x0002

// GATT_Characteristic_Descriptors_Server_Characteristic_Configuration's value
#define GATT_Server_Characteristic_Configuration_Broadcast              0x0001

// GATT_Characteristic_Descriptors_Characteristic_Format's value
#define GATT_Characteristic_Format_rfu      0x00
#define GATT_Characteristic_Format_boolean  0x01
#define GATT_Characteristic_Format_2bit     0x02
#define GATT_Characteristic_Format_nibble   0x03
#define GATT_Characteristic_Format_uint8    0x04
#define GATT_Characteristic_Format_uint12   0x05
#define GATT_Characteristic_Format_uint16   0x06
#define GATT_Characteristic_Format_uint24   0x07
#define GATT_Characteristic_Format_uint32   0x08
#define GATT_Characteristic_Format_uint48   0x09
#define GATT_Characteristic_Format_uint64   0x0a
#define GATT_Characteristic_Format_uint128  0x0b
#define GATT_Characteristic_Format_sint8    0x0c
#define GATT_Characteristic_Format_sint12   0x0d
#define GATT_Characteristic_Format_sint16   0x0e
#define GATT_Characteristic_Format_sint24   0x0f
#define GATT_Characteristic_Format_sint32   0x10
#define GATT_Characteristic_Format_sint48   0x11
#define GATT_Characteristic_Format_sint64   0x12
#define GATT_Characteristic_Format_sint128  0x13
#define GATT_Characteristic_Format_float32  0x14
#define GATT_Characteristic_Format_float64  0x15
#define GATT_Characteristic_Format_SFLOAT   0x16
#define GATT_Characteristic_Format_FLOAT    0x17
#define GATT_Characteristic_Format_duint16  0x18
#define GATT_Characteristic_Format_utf8s    0x19
#define GATT_Characteristic_Format_utf16s   0x1a
#define GATT_Characteristic_Format_struct   0x1b

#define GATT_Characteristic_Unit_unitless                                               0x2700
#define GATT_Characteristic_Unit_length_metre                                           0x2701
#define GATT_Characteristic_Unit_mass_kilogram                                          0x2702
#define GATT_Characteristic_Unit_time_second                                            0x2703
#define GATT_Characteristic_Unit_electric_current_ampere                                0x2704
#define GATT_Characteristic_Unit_thermodynamic_temperature_kelvin                       0x2705
#define GATT_Characteristic_Unit_amount_of_substance_mole                               0x2706
#define GATT_Characteristic_Unit_luminous_intensity_candela                             0x2707
#define GATT_Characteristic_Unit_area_square_metres                                     0x2710
#define GATT_Characteristic_Unit_volume_cubic_metres                                    0x2711
#define GATT_Characteristic_Unit_velocity_metres_per_second                             0x2712
#define GATT_Characteristic_Unit_acceleration_metres_per_second_squared                 0x2713
#define GATT_Characteristic_Unit_wavenumber_reciprocal_metre                            0x2714
#define GATT_Characteristic_Unit_density_kilogram_per_cubic_metre                       0x2715
#define GATT_Characteristic_Unit_surface_density_kilogram_per_square_metre              0x2716
#define GATT_Characteristic_Unit_specific_volume_cubic_metre_per_kilogram               0x2717
#define GATT_Characteristic_Unit_current_density_ampere_per_square_metre                0x2718
#define GATT_Characteristic_Unit_magnetic_field_strength_ampere_per_metre               0x2719
#define GATT_Characteristic_Unit_amount_concentration_mole_per_cubic_metre              0x271a
#define GATT_Characteristic_Unit_mass_concentration_kilogram_per_cubic_metre            0x271b
#define GATT_Characteristic_Unit_luminance_candela_per_square_metre                     0x271c
#define GATT_Characteristic_Unit_refractive_index                                       0x271d
#define GATT_Characteristic_Unit_relative_permeability                                  0x271e
#define GATT_Characteristic_Unit_plane_angle_radian                                     0x2720
#define GATT_Characteristic_Unit_solid_angle_steradian                                  0x2721
#define GATT_Characteristic_Unit_frequency_hertz                                        0x2722
#define GATT_Characteristic_Unit_force_newton                                           0x2723
#define GATT_Characteristic_Unit_pressure_pascal                                        0x2724
#define GATT_Characteristic_Unit_energy_joule                                           0x2725
#define GATT_Characteristic_Unit_power_watt                                             0x2726
#define GATT_Characteristic_Unit_electric_charge_coulomb                                0x2727
#define GATT_Characteristic_Unit_electric_potential_difference_volt                     0x2728
#define GATT_Characteristic_Unit_capacitance_farad                                      0x2729
#define GATT_Characteristic_Unit_electric_resistance_ohm                                0x272a
#define GATT_Characteristic_Unit_electric_conductance_siemens                           0x272b
#define GATT_Characteristic_Unit_magnetic_flex_weber                                    0x272c
#define GATT_Characteristic_Unit_magnetic_flex_density_tesla                            0x272d
#define GATT_Characteristic_Unit_inductance_henry                                       0x272e
#define GATT_Characteristic_Unit_thermodynamic_temperature_degree_celsius               0x272f
#define GATT_Characteristic_Unit_luminous_flux_lumen                                    0x2730
#define GATT_Characteristic_Unit_illuminance_lux                                        0x2731
#define GATT_Characteristic_Unit_activity_referred_to_a_radionuclide_becquerel          0x2732
#define GATT_Characteristic_Unit_absorbed_dose_gray                                     0x2733
#define GATT_Characteristic_Unit_dose_equivalent_sievert                                0x2734
#define GATT_Characteristic_Unit_catalytic_activity_katal                               0x2735
#define GATT_Characteristic_Unit_dynamic_viscosity_pascal_second                        0x2740
#define GATT_Characteristic_Unit_moment_of_force_newton_metre                           0x2741
#define GATT_Characteristic_Unit_surface_tension_newton_per_metre                       0x2742
#define GATT_Characteristic_Unit_angular_velocity_radian_per_second                     0x2743
#define GATT_Characteristic_Unit_angular_acceleration_radian_per_second_squared         0x2744
#define GATT_Characteristic_Unit_heat_flux_density_watt_per_square_metre                0x2745
#define GATT_Characteristic_Unit_heat_capacity_joule_per_kelvin                         0x2746
#define GATT_Characteristic_Unit_specific_heat_capacity_joule_per_kilogram_kelvin       0x2747
#define GATT_Characteristic_Unit_specific_energy_joule_per_kilogram                     0x2748
#define GATT_Characteristic_Unit_thermal_conductivity_watt_per_metre_kelvin             0x2749
#define GATT_Characteristic_Unit_energy_density_joule_per_cubic_metre                   0x274a
#define GATT_Characteristic_Unit_electric_field_strength_volt_per_metre                 0x274b
#define GATT_Characteristic_Unit_electric_charge_density_coulomb_per_cubic_metre        0x274c
#define GATT_Characteristic_Unit_surface_charge_density_coulomb_per_square_metre        0x274d
#define GATT_Characteristic_Unit_electric_flux_density_coulomb_per_square_metre         0x274e
#define GATT_Characteristic_Unit_permittivity_farad_per_metre                           0x274f
#define GATT_Characteristic_Unit_permeability_henry_per_metre                           0x2750
#define GATT_Characteristic_Unit_molar_energy_joule_per_mole                            0x2751
#define GATT_Characteristic_Unit_molar_entropy_joule_per_mole_kelvin                    0x2752
#define GATT_Characteristic_Unit_exposure_coulomb_per_kilogram                          0x2753
#define GATT_Characteristic_Unit_absorbed_dose_rate_gray_per_second                     0x2754
#define GATT_Characteristic_Unit_radiant_intensity_watt_per_steradian                   0x2755
#define GATT_Characteristic_Unit_radiance_watt_per_square_meter_steradian               0x2756
#define GATT_Characteristic_Unit_catalytic_activity_concentration_katal_per_cubic_metre 0x2757
#define GATT_Characteristic_Unit_time_minute                                            0x2760
#define GATT_Characteristic_Unit_time_hour                                              0x2761
#define GATT_Characteristic_Unit_time_day                                               0x2762
#define GATT_Characteristic_Unit_plane_angle_degree                                     0x2763
#define GATT_Characteristic_Unit_plane_angle_minute                                     0x2764
#define GATT_Characteristic_Unit_plane_angle_second                                     0x2765
#define GATT_Characteristic_Unit_area_hectare                                           0x2766
#define GATT_Characteristic_Unit_volume_litre                                           0x2767
#define GATT_Characteristic_Unit_mass_tonne                                             0x2768
#define GATT_Characteristic_Unit_pressure_bar                                           0x2780
#define GATT_Characteristic_Unit_pressure_millimetre_of_mercury                         0x2781
#define GATT_Characteristic_Unit_length_angstrom                                        0x2782
#define GATT_Characteristic_Unit_length_nautical_mile                                   0x2783
#define GATT_Characteristic_Unit_area_barn                                              0x2784
#define GATT_Characteristic_Unit_velocity_knot                                          0x2785
#define GATT_Characteristic_Unit_logarithmic_radio_quantity_neper                       0x2786
#define GATT_Characteristic_Unit_logarithmic_radio_quantity_bel                         0x2787
#define GATT_Characteristic_Unit_length_yard                                            0x27a0
#define GATT_Characteristic_Unit_length_parsec                                          0x27a1
#define GATT_Characteristic_Unit_length_inch                                            0x27a2
#define GATT_Characteristic_Unit_length_foot                                            0x27a3
#define GATT_Characteristic_Unit_length_mile                                            0x27a4
#define GATT_Characteristic_Unit_pressure_pound_force_per_square_inch                   0x27a5
#define GATT_Characteristic_Unit_velocity_kilometre_per_hour                            0x27a6
#define GATT_Characteristic_Unit_velocity_mile_per_hour                                 0x27a7
#define GATT_Characteristic_Unit_angular_velocity_revolution_per_minute                 0x27a8
#define GATT_Characteristic_Unit_energy_gram_calorie                                    0x27a9
#define GATT_Characteristic_Unit_energy_kilogram_calorie                                0x27aa
#define GATT_Characteristic_Unit_energy_kilowatt_hour                                   0x27ab
#define GATT_Characteristic_Unit_thermodynamic_temperature_degree_fahrenheit            0x27ac
#define GATT_Characteristic_Unit_percentage                                             0x27ad
#define GATT_Characteristic_Unit_per_mille                                              0x27ae
#define GATT_Characteristic_Unit_period_beats_per_minute                                0x27af
#define GATT_Characteristic_Unit_electric_charge_ampere_hours                           0x27b0
#define GATT_Characteristic_Unit_mass_density_milligram_per_decilitre                   0x27b1
#define GATT_Characteristic_Unit_mass_density_millimole_per_litre                       0x27b2
#define GATT_Characteristic_Unit_time_year                                              0x27b3
#define GATT_Characteristic_Unit_time_month                                             0x27b4

#define GATT_Characteristic_Namespace_Bluetooth_SIG_Assigned_Numbers                    1

#define GATT_Characteristic_Description_Bluetooth_SIG_unknown       0x0000
#define GATT_Characteristic_Description_Bluetooth_SIG_first         0x0001
#define GATT_Characteristic_Description_Bluetooth_SIG_second        0x0002
#define GATT_Characteristic_Description_Bluetooth_SIG_third         0x0003
//                                                        :
#define GATT_Characteristic_Description_Bluetooth_SIG_two_hundred_and_fifty_fifth    0x00ff
#define GATT_Characteristic_Description_Bluetooth_SIG_front         0x0100
#define GATT_Characteristic_Description_Bluetooth_SIG_back          0x0101
#define GATT_Characteristic_Description_Bluetooth_SIG_top           0x0102
#define GATT_Characteristic_Description_Bluetooth_SIG_bottom        0x0103
#define GATT_Characteristic_Description_Bluetooth_SIG_upper         0x0104
#define GATT_Characteristic_Description_Bluetooth_SIG_lower         0x0105
#define GATT_Characteristic_Description_Bluetooth_SIG_main          0x0106
#define GATT_Characteristic_Description_Bluetooth_SIG_backup        0x0107
#define GATT_Characteristic_Description_Bluetooth_SIG_auxiliary     0x0108
#define GATT_Characteristic_Description_Bluetooth_SIG_supplementary 0x0109
#define GATT_Characteristic_Description_Bluetooth_SIG_flash         0x010A
#define GATT_Characteristic_Description_Bluetooth_SIG_inside        0x010B
#define GATT_Characteristic_Description_Bluetooth_SIG_outside       0x010C
#define GATT_Characteristic_Description_Bluetooth_SIG_left          0x010D
#define GATT_Characteristic_Description_Bluetooth_SIG_right         0x010E
#define GATT_Characteristic_Description_Bluetooth_SIG_internal      0x010F
#define GATT_Characteristic_Description_Bluetooth_SIG_external      0x0110

#define GATT_Characteristic_Appearance_Category_Generic_Phone                   64
#define GATT_Characteristic_Appearance_Category_Generic_Computer                128
#define GATT_Characteristic_Appearance_Category_Generic_Watch                   192
#define GATT_Characteristic_Appearance_Category_Watch_Sports_Watch              193
#define GATT_Characteristic_Appearance_Category_Generic_Clock                   256
#define GATT_Characteristic_Appearance_Category_Generic_Display                 320
#define GATT_Characteristic_Appearance_Category_Generic_Eye_glasses             448
#define GATT_Characteristic_Appearance_Category_Generic_Tag                     512
#define GATT_Characteristic_Appearance_Category_Generic_Keyring                 576
#define GATT_Characteristic_Appearance_Category_Generic_Media_Player            640
#define GATT_Characteristic_Appearance_Category_Generic_Barcode_Scanner         704
#define GATT_Characteristic_Appearance_Category_Generic_Thermometer             768
#define GATT_Characteristic_Appearance_Category_Thermometer_Ear                 769
#define GATT_Characteristic_Appearance_Category_Generic_Heart_rate_Sensor       832
#define GATT_Characteristic_Appearance_Category_Heart_Rate_Sensor_Belt          833
#define GATT_Characteristic_Appearance_Category_Generic_Remote_Control          843
#define GATT_Characteristic_Appearance_Category_Generic_Blood_Pressure          896
#define GATT_Characteristic_Appearance_Category_Blood_Pressure_Arm              897
#define GATT_Characteristic_Appearance_Category_Blood_Pressure_Wrist            898
#define GATT_Characteristic_Appearance_Category_Generic_Human_Interface_Device  960
#define GATT_Characteristic_Appearance_Category_HID_Keyboard                    961
#define GATT_Characteristic_Appearance_Category_HID_Mouse                       962
#define GATT_Characteristic_Appearance_Category_HID_Joystick                    963
#define GATT_Characteristic_Appearance_Category_HID_Gamepad                     964
#define GATT_Characteristic_Appearance_Category_HID_Digitizer_Tablet            965
#define GATT_Characteristic_Appearance_Category_HID_Card_Reader                 966
#define GATT_Characteristic_Appearance_Category_HID_Digital_Pen                 967
#define GATT_Characteristic_Appearance_Category_HID_Barcode_Scanner             968
#define GATT_Characteristic_Appearance_Category_Generic_Glucose_Meter           1024
#define GATT_Characteristic_Appearance_Category_Generic_Running_Walking_Sensor  1088
#define GATT_Characteristic_Appearance_Category_Running_Walking_Sensor_In_Shoe  1089
#define GATT_Characteristic_Appearance_Category_Running_Walking_Sensor_On_Shoe  1090
#define GATT_Characteristic_Appearance_Category_Running_Walking_Sensor_On_Hip   1091
#define GATT_Characteristic_Appearance_Category_Generic_Cycling                 1152
#define GATT_Characteristic_Appearance_Category_Cycling_Computer                1153
#define GATT_Characteristic_Appearance_Category_Cycling_Speed_Sensor            1154
#define GATT_Characteristic_Appearance_Category_Cycling_Cadence_Sensor          1155
#define GATT_Characteristic_Appearance_Category_Cycling_Power_Sensor            1156
#define GATT_Characteristic_Appearance_Category_Cycling_Speed_and_Cadence_Sensor 1157
#define GATT_Characteristic_Appearance_Category_Generic_Pulse_Oximeter          3136
#define GATT_Characteristic_Appearance_Category_Pulse_Oximeter_Fingertip        3137
#define GATT_Characteristic_Appearance_Category_Pulse_Oximeter_Wrist_Worn       3138
#define GATT_Characteristic_Appearance_Category_Generic_Weight_Scale            3200
#define GATT_Characteristic_Appearance_Category_Generic_Outdoor_Sports_Activity 5184
#define GATT_Characteristic_Appearance_Category_Location_Disp_lay_Device        5185
#define GATT_Characteristic_Appearance_Category_Location_and_Navigation_Display_Device 5186
#define GATT_Characteristic_Appearance_Category_Location_Pod                    5187
#define GATT_Characteristic_Appearance_Category_Location_and_Navigation_Pod     5188


union bt_gatt_Attribute_Value {
    uint8_t MaxArea[20];

    struct {
        uint8_t UUID[16];                    // 2 or 16bytes
    }       Primary_Service;

    struct {
        uint8_t UUID[16];                    // 2 or 16bytes
    }       Secondary_Service;

    struct {
        uint8_t Included_Service_Attribute_Handle[2];
        uint8_t End_Group_Handle[2];
        uint8_t Service_UUID[16];            // 2 or 16bytes
    }       Include;

    struct {
        uint8_t Characteristic_Properties;
        uint8_t Characteristic_Value_Handle[2];
        uint8_t Characteristic_UUID[16];    // 2 or 16bytes
    }       Characteristic;

    struct {
        uint8_t Characteristic_Value[20];
    }       CharacteristicValue;

    struct {
        uint8_t Characteristic_Extended_Properties_Bit_Field[2];
    }       Characteristic_Extended_Properties;

    struct {
        uint8_t Characteristic_User_Description_UTF8_String[20];
    }       Characteristic_User_Description;

    struct {
        uint8_t Characteristic_Configuration_Bits[2];
    }       Client_Characteristic_Configuration;

    struct {
        uint8_t Characteristic_Configuration_Bits[2];
    }       Server_Characteristic_Configuration;

    struct {
        uint8_t Format;
        uint8_t Exponent;
        uint8_t Unit[2];
        uint8_t Name_Space;
        uint8_t Description[2];
    }       Characteristic_Format;

    struct {
        uint8_t Attribute_Handles[10][2];
    }       Characteristic_Aggregate_Format;

}  __attribute__ ((packed)) ;

#ifdef DELETE
struct bt_gatt_definition2 {
    uint8_t                         Attribute_Handle[2];
    uint8_t                         Attribute_Types[2];
    union bt_gatt_Attribute_Value   Attribute_Value;
//  uint8_t    *permissions;
}  __attribute__ ((packed));

struct bt_gatt_definition16 {
    uint8_t                         Attribute_Handle[2];
    uint8_t                         Attribute_Types[16];
    union bt_gatt_Attribute_Value   Attribute_Value;
//  uint8_t    *permissions;
}  __attribute__ ((packed));
#endif

#endif /* BT_SPEC_GATT_H_ */
