//
//  bt_spec_smp.h
//
//  Bluetooth Protocol Stack - SM(Security Manager) Specification definition
//  Copyright (C) 2014-2017 Toyohiko Togashi tog001@nifty.com
//
//
//  This program is free software; you can redistribute it and/or modify it under the terms of the
//  GNU General Public License as published by the Free Software Foundation; either version 3
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
//  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with this program.
//  If not, see <http://www.gnu.org/licenses/>
//
//
//  Reference:
//      Bluetooth SIG (www.bluetooth.com)
//          BLUETOOTH SPECIFICATION Version 4.0
//          [Vol 3] Core System Package [Host volume]
//           Part H: SECURITY MANAGER SPECIFICATION
//
//  Update history
//  ---------- ----- -------------------------------------------
//  2014.02.23 v0.0  First cording
//  2014.03.05 v0.2  Commit
//  2017.01.18 v0.3f Appended a authentication function
//

#ifndef BT_SPEC_SMP_H_
#define BT_SPEC_SMP_H_

#include <stdint.h>

struct bt_smp_PDU_Format {
    uint8_t         Code;
    union {
        uint8_t     Data[22];
        struct {
            uint8_t     IO_Capability;
            uint8_t     OOB_data_flag;
            uint8_t     AuthReq;
            uint8_t     Maximum_Encryption_Key_Size;
            uint8_t     Initiator_Key_Distribution;
            uint8_t     Responder_Key_Distribution;
        }           Pairing_Request;
        struct {
            uint8_t     IO_Capability;
            uint8_t     OOB_data_flag;
            uint8_t     AuthReq;
            uint8_t     Maximum_Encryption_Key_Size;
            uint8_t     Initiator_Key_Distribution;
            uint8_t     Responder_Key_Distribution;
        }           Pairing_Response;
        struct {
            uint8_t     Confirm_value[16];  // Little endianness
        }           Pairing_Confirm;
        struct {
            uint8_t     Random_value[16];
        }           Pairing_Random;
        struct {
            uint8_t     Reason;
        }           Pairing_Failed;
        struct {
            uint8_t     Long_Term_Key[16];
        }           Encryption_Information;
        struct {
            uint8_t     EDIV[2];
            uint8_t     Rand[8];
        }           Master_Identification;
        struct {
            uint8_t     Identity_Resolving_Key[16];
        }           Identity_Information;
        struct {
            uint8_t     AddrType;
            uint8_t     BD_ADDR[6];
        }           Identity_Address_Information;
        struct {
            uint8_t     Signature_Key[16];
        }           Signing_Information;
        struct {
            uint8_t     AuthReq;
        }           Security_Request;
    };
};

#define SMP_PDU_Format_HeadSize                 1       // Opcode

#define SMP_Code_Pairing_Request                0x01
#define SMP_Code_Pairing_Response               0x02
#define SMP_Code_Pairing_Confirm                0x03
#define SMP_Code_Pairing_Random                 0x04
#define SMP_Code_Pairing_Failed                 0x05
#define SMP_Code_Encryption_Information         0x06    // Key Distribution EncKey bit
#define SMP_Code_Master_Identification          0x07    // Key Distribution EncKey bit
#define SMP_Code_Identity_Information           0x08    // Key Distribution IdKey bit
#define SMP_Code_Identity_Address_Information   0x09    // Key Distribution IdKey bit
#define SMP_Code_Signing_Information            0x0A    // Key Distribution Sign bit
#define SMP_Code_Security_Request               0x0B

#define SMP_IO_Capability_DisplayOnly           0x00
#define SMP_IO_Capability_DisplayYesNo          0x01
#define SMP_IO_Capability_KeyboardOnly          0x02
#define SMP_IO_Capability_NoInputNoOutput       0x03
#define SMP_IO_Capability_KeyboardDisplay       0x04

#define SMP_OOB_data_flag_Authentication_data_not_present                   0x00
#define SMP_OOB_data_flag_Authentication_data_from_remote_device_present    0x01

#define SMP_AuthReq_No_Bonding                  0b000
#define SMP_AuthReq_Bonding                     0b001
#define SMP_AuthReq_MITM                        0b100

#define SMP_Reason_Passkey_Entry_Failed         0x01
#define SMP_Reason_OOB_Not_Available            0x02
#define SMP_Reason_Authentication_Requirements  0x03
#define SMP_Reason_Confirm_Value_Failed         0x04
#define SMP_Reason_Pairing_Not_Supported        0x05
#define SMP_Reason_Encryption_Key_Size          0x06
#define SMP_Reason_Command_Not_Supported        0x07
#define SMP_Reason_Unspecified_Reason           0x08
#define SMP_Reason_Repeated_Attempts            0x09
#define SMP_Reason_Invalid_Parameters           0x0a

#define SMP_Key_Distribution_EncKey             0b001   // Encryption Information and Master Identification command (LTK,EDIV,Rand)
#define SMP_Key_Distribution_IdKey              0b010   // Identity Information and Identity Address Information command (IRK)
#define SMP_Key_Distribution_Sign               0b100   // Signing Information command (CSRK)

#define SMP_Timeout                             30      // ADD 2017.01.18
#define SMP_Maximum_Encryption_Key_Size         16      // ADD 2017.01.18
#define SMP_Minimum_Encryption_Key_Size         7       // ADD 2017.01.18

#endif /* BT_SPEC_SMP_H_ */
