//
//  bt_spec_hci.h
//
//  Bluetooth Protocol Stack - HCI(Host Controller Interface) Specification definition
//  Copyright (C) 2013-2019 Toyohiko Togashi tog001@nifty.com
//
//
//  This program is free software; you can redistribute it and/or modify it under the terms of the
//  GNU General Public License as published by the Free Software Foundation; either version 3
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
//  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with this program.
//  If not, see <http://www.gnu.org/licenses/>
//
//
//  Reference:
//      Bluetooth SIG (www.bluetooth.com)
//          BLUETOOTH SPECIFICATION Version 4.0
//          Volume 2: Core System Package [Controller volume]
//          Part E: Host Controller Interface Functional Specification
//
//  Update history
//  ---------- ----- -------------------------------------------
//  2013.01.21 v0.0  First cording
//  2013.04.21 v0.1  Commit
//  2013.12.19 v0.1a Bug fix
//  2014.02.23 v0.1a Supported the Security Manager
//  2015.02.06 v0.3  Appended: Send and recieve messages
//  2015.02.11 v0.3a Appended: Initial byte string macros
//  2015.02.20 v0.3b UUID conver macro
//  2016.04.21 v0.3d Bug fix (BD_ADDR length)
//  2017.01.18 v0.3e Appended: Address_Type
//  2019.03.09 v0.4d Bug fix (ACL-U packet size)
//
#ifndef BT_SPEC_HCI_H_
#define BT_SPEC_HCI_H_

#include <stdint.h>

//
//  Command
//
struct bt_hci_Command_Packet {
    uint8_t     Op_Code[2];
    uint8_t     Parameter_Total_Length;
    union {
        // 7.1 Link Control commands
        struct {
            uint8_t     LAP[3];
            uint8_t     Inquiry_Length;
            uint8_t     Num_Responses;
        }           Inquiry;

        struct {
            uint8_t     Status;
        }           Inquiry_Cancel;

        struct {
            uint8_t     Max_Period_Length[2];
            uint8_t     Min_Period_Length[2];
            uint8_t     LAP[3];
            uint8_t     Inquiry_Length;
            uint8_t     Num_Responses;
        }           Periodic_Inquiry_Mode;

        struct {
            uint8_t     Status;
        }           Exit_Periodic_Inquiry_Mode;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Packet_Type[2];
            uint8_t     Page_Scan_Repetition_Mode;
            uint8_t     Reserved;
            uint8_t     Clock_Offset[2];
            uint8_t     Allow_Role_Switch;
        }           Create_Connection;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Reason;
        }           Disconnect;

        struct {
            uint8_t     BD_ADDR[6];
        }           Create_Connection_Cancel;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Role;
        }           Accept_Connection_Request;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Reason;
        }           Reject_Connection_Request;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Link_Key[16];
        }           Link_Key_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           Link_Key_Request_Negative_Reply;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     PIN_Code_Length;
            uint8_t     PIN_Code[16];
        }           PIN_Code_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           PIN_Code_Request_Negative_Reply;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Packet_Type;
        }           Change_Connection_Packet_Type;

        struct {
            uint8_t     Connection_Handle[2];
        }           Authentication_Requested;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Encryption_Enable;
        }           Set_Connection_Encryption;

        struct {
            uint8_t     Connection_Handle[2];
        }           Change_Connection_Link_Key;

        struct {
            uint8_t     Key_Flag;
        }           Master_Link_Key;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Page_Scan_Repetition_Mode;
            uint8_t     Reserved;
            uint8_t     Clock_Offset[2];
        }           Remote_Name_Request;

        struct {
            uint8_t     Status;
            uint8_t     BD_ADDR[6];
        }           Remote_Name_Request_Cancel;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Remote_Supported_Features;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Page_Number;
        }           Read_Remote_Extended_Features;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Remote_Version_Information;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Clock_Offset;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_LMP_Handle;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Transmit_Bandwidth[4];
            uint8_t     Receive_Bandwidth[4];
            uint8_t     Max_Latency[2];
            uint8_t     Voice_Setting[2];
            uint8_t     Retransmission_Effort;
            uint8_t     Packet_Type[2];
        }           Setup_Synchronous_Connection;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Transmit_Bandwidth[4];
            uint8_t     Receive_Bandwidth[4];
            uint8_t     Max_Latency[2];
            uint8_t     Content_Format[2];
            uint8_t     Retransmission_Effort;
            uint8_t     Packet_Type[2];
        }           Accept_Synchronous_Connection_Request;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Reason;
        }           Reject_Synchronous_Connection_Request;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     IO_Capability;
            uint8_t     OOB_Data_Present;
            uint8_t     Authentication_Requirements;
        }           IO_Capability_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           User_Confirmation_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           User_Confirmation_Request_Negative_Reply;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Numeric_Value[4];
        }           User_Passkey_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           User_Passkey_Request_Negative_Reply;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     C[16];
            uint8_t     R[16];
        }           Remote_OOB_Data_Request_Reply;

        struct {
            uint8_t     BD_ADDR[6];
        }           Remote_OOB_Data_Request_Negative_Reply;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Reason;
        }           IO_Capability_Request_Negative_Reply;

        // 7.2 Link Policy Commands
        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Hold_Mode_Max_Interval[2];
            uint8_t     Hold_Mode_Min_Interval[2];
        }           Hold_Mode;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Sniff_Max_Interval[2];
            uint8_t     Sniff_Min_Interval[2];
            uint8_t     Sniff_Attempt[2];
            uint8_t     Sniff_Timeout[2];
        }           Sniff_Mode;

        struct {
            uint8_t     Connection_Handle[2];
        }           Exit_Sniff_Mode;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Beacon_Max_Interval[2];
            uint8_t     Beacon_Min_Interval[2];
        }           Park_State;

        struct {
            uint8_t     Connection_Handle[2];
        }           Exit_Park_State;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Flags;
            uint8_t     Service_Type;
            uint8_t     Token_Rate[4];
            uint8_t     Peak_Bandwidth[4];
            uint8_t     Latency[4];
            uint8_t     Delay_Variation[4];
        }           QoS_Setup;

        struct {
            uint8_t     Connection_Handle[2];
        }           Role_Discovery;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Role;
        }           Switch_Role;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Link_Policy_Settings;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Link_Policy_Settings[2];
        }           Write_Link_Policy_Settings;

        struct {
            uint8_t     Link_Policy_Settings[2];
        }           Write_Default_Link_Policy_Settings;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Flags;
            uint8_t     Flow_direction;
            uint8_t     Service_Type;
            uint8_t     Token_Rate[4];
            uint8_t     Token_Bucket_Size[4];
            uint8_t     Peak_Bandwidth[4];
            uint8_t     Access_Latency[4];
        }           Flow_Specification;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Maximum_Latency[2];
            uint8_t     Minimum_Remote_Timeout[2];
            uint8_t     Minimum_Local_Timeout[2];
        }           Sniff_Subrating;

        // 7.3 Controller & Baseband Commands
        struct {
            uint8_t     Event_Mask[8];
        }           Set_Event_Mask;

        struct {
            uint8_t     Filter_Type;
            uint8_t     Inquiry_Result_Filter_Condition_Type;
            uint8_t     Connection_Setup_Filter_Condition_Type;
        }           Set_Event_Filter;

        struct {
            uint8_t     Connection_Handle[2];
        }           Flush;

        struct {
            uint8_t     PIN_Type;
        }           Write_PIN_Type;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Read_All_Flag;
        }           Read_Stored_Link_Key;

        struct {
            uint8_t     Num_Keys_To_Write;
            uint8_t     BD_ADDR[6];             // *Num_Keys_To_Write
            uint8_t     Link_Key[16];           // *Num_Keys_To_Write
        }           Write_Stored_Link_Key;

        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Delete_All_Flag;
        }           Delete_Stored_Link_Key;

        struct {
            uint8_t     Local_Name[248];
        }           Write_Local_Name;

        struct {
            uint8_t     Conn_Accept_Timeout;
        }           Write_Connection_Accept_Timeout;

        struct {
            uint8_t     Page_Timeout[2];
            uint8_t     Delete_All_Flag;
        }           Write_Page_Timeout;

        struct {
            uint8_t     Scan_Enable;
        }           Write_Scan_Enable;

        struct {
            uint8_t     Page_Scan_Interval[2];
            uint8_t     Page_Scan_Window[2];
        }           Write_Page_Scan_Activity;

        struct {
            uint8_t     Inquiry_Scan_Interval[2];
            uint8_t     Inquiry_Scan_Window[2];
        }           Write_Inquiry_Scan_Activity;

        struct {
            uint8_t     Authentication_Enable;
        }           Write_Authentication_Enable;

        struct {
            uint8_t     Class_of_Device[3];
        }           Write_Class_of_Device;

        struct {
            uint8_t     Voice_Setting[2];
        }           Write_Voice_Setting;

        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Automatic_Flush_Timeout;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Flush_Timeout[2];
        }           Write_Automatic_Flush_Timeout;

        //---

        struct {
            uint8_t     Simple_Pairing_Mode;
        }           Write_Simple_Pairing_Mode;

        //---

        struct {
            uint8_t     LE_Supported_Host;
            uint8_t     Simultaneous_LE_Host;
        }           Write_LE_Host_Supported;

        // 7.4 Informational Parameters
        struct {
            uint8_t     Page_number;
        }           Read_Local_Extended_Features;

        // 7.5 Status Parameters
        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Failed_Contact_Counter;
        struct {
            uint8_t     Connection_Handle[2];
        }           Reset_Failed_Contact_Counter;
        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Link_Quality;
        struct {
            uint8_t     Connection_Handle[2];
        }           Read_RSSI;
        struct {
            uint8_t     Connection_Handle[2];
        }           Read_AFH_Channel_Map;
        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Which_Clock;
        }           Read_Clock;
        struct {
            uint8_t     Connection_Handle[2];
        }           Read_Encryption_Key_Size;

        // 7.6 Testing commands
        struct {
            uint8_t     Loopback_Mode;
        }           Write_Loopback_Mode;
        struct {
            uint8_t     Simple_Pairing_Debug_Mode;
        }           Write_Simple_Pairing_Debug_Mode;

        // 7.7 LE Controller Commands
        struct {
            uint8_t     LE_Event_Mask[8];
        }           LE_Set_Event_Mask;

        struct {
            uint8_t     Random_Addess[6];
        }           LE_Set_Random_Address;

        struct {
            uint8_t     Advertising_Interval_Min[2];
            uint8_t     Advertising_Interval_Max[2];
            uint8_t     Advertising_Type;
            uint8_t     Own_Address_Type;
            uint8_t     Direct_Address_Type;
            uint8_t     Direct_Address[6];
            uint8_t     Advertising_Channel_Map;
            uint8_t     Advertising_Filter_Policy;
        }           LE_Set_Advertising_Parameters;

        struct {
            uint8_t     Advertising_Data_Length;
            uint8_t     Advertising_Data[31];
        }           LE_Set_Advertising_Data;

        struct {
            uint8_t     Scan_Response_Data_Length;
            uint8_t     Scan_Response_Data[31];
        }           LE_Set_Scan_Response_Data;

        struct {
            uint8_t     Advertising_Enable;
        }           LE_Set_Advertise_Enable;

        struct {
            uint8_t     LE_Scan_Type;
            uint8_t     LE_Scan_Interval[2];
            uint8_t     LE_Scan_Window[2];
            uint8_t     Own_Address_Type;
            uint8_t     Scanning_Filter_Policy;
        }           LE_Set_Scan_Parameters;

        struct {
            uint8_t     Synchronous_Flow_Control_Enable;
        }           LE_Set_Scan_Enable;

        struct {
            uint8_t     LE_Scan_Interval[2];
            uint8_t     LE_Scan_Window[2];
            uint8_t     Initiator_Filter_Policy;
            uint8_t     Peer_Address_Type;
            uint8_t     Peer_Address[6];
            uint8_t     Own_Address_Type;
            uint8_t     Conn_Interval_Min[2];
            uint8_t     Conn_Interval_Max[2];
            uint8_t     Conn_Latency[2];
            uint8_t     Supervision_Timeout[2];
            uint8_t     Minimum_CE_Length[2];
            uint8_t     Maximum_CE_Length[2];
        }           LE_Create_Connection;

        struct {
            uint8_t     Address_Type;
            uint8_t     Address[6];
        }           LE_Add_Device_To_White_List;

        struct {
            uint8_t     Address_Type;
            uint8_t     Address[6];
        }           LE_Remove_Device_From_White_List;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Conn_Interval_Min[2];
            uint8_t     Conn_Interval_Max[2];
            uint8_t     Conn_Latency[2];
            uint8_t     Supervision_Timeout[2];
            uint8_t     Minimum_CE_Length[2];
            uint8_t     Maximum_CE_Length[2];
        }           LE_Connection_Update;

        struct {
            uint8_t     Channel_Map[5];
        }           LE_Set_Host_Channel_Classification;

        struct {
            uint8_t     Connection_Handle[2];
        }           LE_Read_Channel_Map;

        struct {
            uint8_t     Connection_Handle[2];
        }           LE_Read_Remote_Used_Features;

                                        // ADD START 2014.02.23
        struct {
            uint8_t     Key[16];
            uint8_t     Plaintext_Data[16];
        }           LE_Encrypt;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Random_Number[8];
            uint8_t     Encrypted_Diversifier[2];
            uint8_t     Long_Term_Key[16];
        }           LE_Start_Encryption;

        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Long_Term_Key[16];
        }           LE_Long_Term_Key_Request_Reply;

        struct {
            uint8_t     Connection_Handle[2];
        }           LE_Long_Term_Key_Requested_Negative_Reply;
                                        // ADD END 2014.02.23
        struct {
            uint8_t     RX_Frequency;
        }           LE_Receiver_Test;

    }            Parameter;
}  __attribute__ ((packed)) ;

#define HCI_Command_Packet_HeadSize 3        // Op_Code + Parameter_Total_Length
#define HCI_OPCODE_OGF_SHIFT        10
#define HCI_OPCODE_OGF_MASK         0b1111110000000000
#define HCI_OPCODE_OCF_MASK         0b0000001111111111
#define HCI_OPCODE(ogf,ocf)         (ogf<<HCI_OPCODE_OGF_SHIFT | ocf)

// 7.1 LINK CONTROL COMMANDS
#define HCI_Nop                                         HCI_OPCODE(0x00,0x000)
#define HCI_Inquiry                                     HCI_OPCODE(0x01,0x001)
#define HCI_Inquiry_Cancel                              HCI_OPCODE(0x01,0x002)
#define HCI_Periodic_Inquiry_Mode                       HCI_OPCODE(0x01,0x003)
#define HCI_Exit_Periodic_Inquiry_Mode                  HCI_OPCODE(0x01,0x004)
#define HCI_Create_Connection                           HCI_OPCODE(0x01,0x005)
#define HCI_Disconnect                                  HCI_OPCODE(0x01,0x006)
#define HCI_Create_Connection_Cancel                    HCI_OPCODE(0x01,0x008)
#define HCI_Accept_Connection_Request                   HCI_OPCODE(0x01,0x009)
#define HCI_Reject_Connection_Request                   HCI_OPCODE(0x01,0x00a)
#define HCI_Link_Key_Request_Reply                      HCI_OPCODE(0x01,0x00b)
#define HCI_Link_Key_Request_Negative_Reply             HCI_OPCODE(0x01,0x00c)
#define HCI_PIN_Code_Request_Reply                      HCI_OPCODE(0x01,0x00d)
#define HCI_PIN_Code_Request_Negative_Reply             HCI_OPCODE(0x01,0x00e)
#define HCI_Change_Connection_Packet_Type               HCI_OPCODE(0x01,0x00f)
#define HCI_Authentication_Requested                    HCI_OPCODE(0x01,0x011)
#define HCI_Set_Connection_Encryption                   HCI_OPCODE(0x01,0x013)
#define HCI_Change_Connection_Link_Key                  HCI_OPCODE(0x01,0x015)
#define HCI_Master_Link_Key                             HCI_OPCODE(0x01,0x017)
#define HCI_Remote_Name_Request                         HCI_OPCODE(0x01,0x019)
#define HCI_Remote_Name_Request_Cancel                  HCI_OPCODE(0x01,0x01a)
#define HCI_Read_Remote_Supported_Features              HCI_OPCODE(0x01,0x01b)
#define HCI_Read_Remote_Extended_Features               HCI_OPCODE(0x01,0x01c)
#define HCI_Read_Remote_Version_Information             HCI_OPCODE(0x01,0x01d)
#define HCI_Read_Clock_Offset                           HCI_OPCODE(0x01,0x01f)
#define HCI_Read_LMP_Handle                             HCI_OPCODE(0x01,0x020)
#define HCI_Setup_Synchronous_Connection                HCI_OPCODE(0x01,0x028)
#define HCI_Accept_Synchronous_Connection_Request       HCI_OPCODE(0x01,0x029)
#define HCI_Reject_Synchronous_Connection_Request       HCI_OPCODE(0x01,0x02a)
#define HCI_IO_Capability_Request_Reply                 HCI_OPCODE(0x01,0x02b)
#define HCI_User_Confirmation_Request_Reply             HCI_OPCODE(0x01,0x02c)
#define HCI_User_Confirmation_Request_Negative_Reply    HCI_OPCODE(0x01,0x02d)
#define HCI_User_Passkey_Request_Reply                  HCI_OPCODE(0x01,0x02e)
#define HCI_User_Passkey_Request_Negative_Reply         HCI_OPCODE(0x01,0x02f)
#define HCI_Remote_OOB_Data_Request_Reply               HCI_OPCODE(0x01,0x030)
#define HCI_Remote_OOB_Data_Request_Negative_Reply      HCI_OPCODE(0x01,0x033)
#define HCI_IO_Capability_Request_Negative_Reply        HCI_OPCODE(0x01,0x034)
#define HCI_Create_Physical_Link                        HCI_OPCODE(0x01,0x035)
#define HCI_Accept_Physical_Link                        HCI_OPCODE(0x01,0x036)
#define HCI_Disconnect_Physical_Link                    HCI_OPCODE(0x01,0x037)
#define HCI_Create_Logical_Link                         HCI_OPCODE(0x01,0x038)
#define HCI_Accept_Logical_Link                         HCI_OPCODE(0x01,0x039)
#define HCI_Disconnect_Logical_Link                     HCI_OPCODE(0x01,0x03a)
#define HCI_Logical_Link_Cancel                         HCI_OPCODE(0x01,0x03b)
#define HCI_Flow_Spec_Modify                            HCI_OPCODE(0x01,0x03c)

// 7.2 LINK POLICY COMMANDS
#define HCI_Hold_Mode                                   HCI_OPCODE(0x02,0x001)
#define HCI_Sniff_Mode                                  HCI_OPCODE(0x02,0x003)
#define HCI_Exit_Sniff_Mode                             HCI_OPCODE(0x02,0x004)
#define HCI_Park_State                                  HCI_OPCODE(0x02,0x005)
#define HCI_Exit_Park_State                             HCI_OPCODE(0x02,0x006)
#define HCI_QoS_Setup                                   HCI_OPCODE(0x02,0x007)
#define HCI_Role_Discovery                              HCI_OPCODE(0x02,0x009)
#define HCI_Switch_Role                                 HCI_OPCODE(0x02,0x00b)
#define HCI_Read_Link_Policy_Settings                   HCI_OPCODE(0x02,0x00c)
#define HCI_Write_Link_Policy_Settings                  HCI_OPCODE(0x02,0x00d)
#define HCI_Read_Default_Link_Policy_Settings           HCI_OPCODE(0x02,0x00e)
#define HCI_Write_Default_Link_Policy_Settings          HCI_OPCODE(0x02,0x00f)
#define HCI_Flow_Specification                          HCI_OPCODE(0x02,0x010)
#define HCI_Sniff_Subrating                             HCI_OPCODE(0x02,0x011)

// 7.3 CONTROLLER & BASEBAND COMMANDS
#define HCI_Set_Event_Mask                              HCI_OPCODE(0x03,0x001)
#define HCI_Reset                                       HCI_OPCODE(0x03,0x003)
#define HCI_Set_Event_Filter                            HCI_OPCODE(0x03,0x005)
#define HCI_Flush                                       HCI_OPCODE(0x03,0x008)
#define HCI_Read_PIN_Type                               HCI_OPCODE(0x03,0x009)
#define HCI_Write_PIN_Type                              HCI_OPCODE(0x03,0x00a)
#define HCI_Create_New_Unit_Key                         HCI_OPCODE(0x03,0x00b)
#define HCI_Read_Stored_Link_Key                        HCI_OPCODE(0x03,0x00d)
#define HCI_Write_Stored_Link_Key                       HCI_OPCODE(0x03,0x011)
#define HCI_Delete_Stored_Link_Key                      HCI_OPCODE(0x03,0x012)
#define HCI_Write_Local_Name                            HCI_OPCODE(0x03,0x013)
#define HCI_Read_Local_Name                             HCI_OPCODE(0x03,0x014)
#define HCI_Read_Connection_Accept_Timeout              HCI_OPCODE(0x03,0x015)
#define HCI_Write_Connection_Accept_Timeout             HCI_OPCODE(0x03,0x016)
#define HCI_Read_Page_Timeout                           HCI_OPCODE(0x03,0x017)
#define HCI_Write_Page_Timeout                          HCI_OPCODE(0x03,0x018)
#define HCI_Read_Scan_Enable                            HCI_OPCODE(0x03,0x019)
#define HCI_Write_Scan_Enable                           HCI_OPCODE(0x03,0x01a)
#define HCI_Read_Page_Scan_Activity                     HCI_OPCODE(0x03,0x01b)
#define HCI_Write_Page_Scan_Activity                    HCI_OPCODE(0x03,0x01c)
#define HCI_Read_Inquiry_Scan_Activity                  HCI_OPCODE(0x03,0x01d)
#define HCI_Write_Inquiry_Scan_Activity                 HCI_OPCODE(0x03,0x01e)
#define HCI_Read_Authentication_Enable                  HCI_OPCODE(0x03,0x01f)
#define HCI_Write_Authentication_Enable                 HCI_OPCODE(0x03,0x020)
#define HCI_Read_Class_of_Device                        HCI_OPCODE(0x03,0x023)
#define HCI_Write_Class_of_Device                       HCI_OPCODE(0x03,0x024)
#define HCI_Read_Voice_Setting                          HCI_OPCODE(0x03,0x025)
#define HCI_Write_Voice_Setting                         HCI_OPCODE(0x03,0x026)
#define HCI_Read_Automatic_Flush_Timeout                HCI_OPCODE(0x03,0x027)
#define HCI_Write_Automatic_Flush_Timeout               HCI_OPCODE(0x03,0x028)
#define HCI_Read_Num_Broadcast_Retransmissions          HCI_OPCODE(0x03,0x029)
#define HCI_Write_Num_Broadcast_Retransmissions         HCI_OPCODE(0x03,0x02a)
#define HCI_Read_Hold_Mode_Activity                     HCI_OPCODE(0x03,0x02b)
#define HCI_Write_Hold_Mode_Activity                    HCI_OPCODE(0x03,0x02c)
#define HCI_Read_Transmit_Power_Level                   HCI_OPCODE(0x03,0x02d)
#define HCI_Read_Synchronous_Flow_Control_Enable        HCI_OPCODE(0x03,0x02e)
#define HCI_Write_Synchronous_Flow_Control_Enable       HCI_OPCODE(0x03,0x02f)
#define HCI_Set_Controller_To_Host_Flow_Control         HCI_OPCODE(0x03,0x031)
#define HCI_Host_Buffer_Size                            HCI_OPCODE(0x03,0x033)
#define HCI_Host_Number_Of_Completed_Packets            HCI_OPCODE(0x03,0x035)
#define HCI_Read_Link_Supervision_Timeout               HCI_OPCODE(0x03,0x036)
#define HCI_Write_Link_Supervision_Timeout              HCI_OPCODE(0x03,0x037)
#define HCI_Read_Number_Of_Supported_IAC                HCI_OPCODE(0x03,0x038)
#define HCI_Read_Current_IAC_LAP                        HCI_OPCODE(0x03,0x039)
#define HCI_Write_Current_IAC_LAP                       HCI_OPCODE(0x03,0x03a)
#define HCI_Set_AFH_Host_Channel_Classification         HCI_OPCODE(0x03,0x03f)
#define HCI_Read_Inquiry_Scan_Type                      HCI_OPCODE(0x03,0x042)
#define HCI_Write_Inquiry_Scan_Type                     HCI_OPCODE(0x03,0x043)
#define HCI_Read_Inquiry_Mode                           HCI_OPCODE(0x03,0x044)
#define HCI_Write_Inquiry_Mode                          HCI_OPCODE(0x03,0x045)
#define HCI_Read_Page_Scan_Type                         HCI_OPCODE(0x03,0x046)
#define HCI_Write_Page_Scan_Type                        HCI_OPCODE(0x03,0x047)
#define HCI_Read_AFH_Channel_Assessment_Mode            HCI_OPCODE(0x03,0x048)
#define HCI_Write_AFH_Channel_Assessment_Mode           HCI_OPCODE(0x03,0x049)
#define HCI_Read_Extended_Inquiry_Response              HCI_OPCODE(0x03,0x051)
#define HCI_Write_Extended_Inquiry_Response             HCI_OPCODE(0x03,0x052)
#define HCI_Refresh_Encryption_Key                      HCI_OPCODE(0x03,0x053)
#define HCI_Read_Simple_Pairing_Mode                    HCI_OPCODE(0x03,0x055)
#define HCI_Write_Simple_Pairing_Mode                   HCI_OPCODE(0x03,0x056)
#define HCI_Read_Local_OOB_Data                         HCI_OPCODE(0x03,0x057)
#define HCI_Read_Inquiry_Response_Transmit_Power_Level  HCI_OPCODE(0x03,0x058)
#define HCI_Write_Inquiry_Transmit_Power_Level          HCI_OPCODE(0x03,0x059)
#define HCI_Send_Keypress_Notification                  HCI_OPCODE(0x03,0x060)
#define HCI_Read_Default_Erroneous_Data_Reporting       HCI_OPCODE(0x03,0x05a)
#define HCI_Write_Default_Erroneous_Data_Reporting      HCI_OPCODE(0x03,0x05b)
#define HCI_Enhanced_Flush                              HCI_OPCODE(0x03,0x05f)
#define HCI_Read_Logical_Link_Accept_Timeout            HCI_OPCODE(0x03,0x061)
#define HCI_Write_Logical_Link_Accept_Timeout           HCI_OPCODE(0x03,0x062)
#define HCI_Set_Event_Mask_Page_2                       HCI_OPCODE(0x03,0x063)    // UPD 2013.12.19
#define HCI_Read_Location_Data                          HCI_OPCODE(0x03,0x064)
#define HCI_Write_Location_Data                         HCI_OPCODE(0x03,0x065)
#define HCI_Read_Flow_Control_Mode                      HCI_OPCODE(0x03,0x066)
#define HCI_Write_Flow_Control_Mode                     HCI_OPCODE(0x03,0x067)
#define HCI_Read_Enhanced_Transmit_Power_Level          HCI_OPCODE(0x03,0x068)
#define HCI_Read_Best_Effort_Flush_Timeout              HCI_OPCODE(0x03,0x069)
#define HCI_Write_Best_Effort_Flush_Timeout             HCI_OPCODE(0x03,0x06a)
#define HCI_Short_Range_Mode                            HCI_OPCODE(0x03,0x06b)
#define HCI_Read_LE_Host_Supported                      HCI_OPCODE(0x03,0x06c)
#define HCI_Write_LE_Host_Supported                     HCI_OPCODE(0x03,0x06d)

// 7.4 INFORMATIONAL PARAMETERS
#define HCI_Read_Local_Version_Information              HCI_OPCODE(0x04,0x001)
#define HCI_Read_Local_Supported_Commands               HCI_OPCODE(0x04,0x002)
#define HCI_Read_Local_Supported_Features               HCI_OPCODE(0x04,0x003)
#define HCI_Read_Local_Extended_Features                HCI_OPCODE(0x04,0x004)
#define HCI_Read_Buffer_Size                            HCI_OPCODE(0x04,0x005)
#define HCI_Read_BD_ADDR                                HCI_OPCODE(0x04,0x009)

// 7.5 STATUS PARAMETERS
#define HCI_Read_Failed_Contact_Counter                 HCI_OPCODE(0x05,0x001)
#define HCI_Reset_Failed_Contact_Counter                HCI_OPCODE(0x05,0x002)
#define HCI_Read_Link_Quality                           HCI_OPCODE(0x05,0x003)
#define HCI_Read_RSSI                                   HCI_OPCODE(0x05,0x005)
#define HCI_Read_AFH_Channel_Map                        HCI_OPCODE(0x05,0x006)
#define HCI_Read_Clock                                  HCI_OPCODE(0x05,0x007)
#define HCI_Read_Encryption_Key_Size                    HCI_OPCODE(0x05,0x008)
#define HCI_Read_Local_AMP_Info                         HCI_OPCODE(0x05,0x009)
#define HCI_Read_Local_AMP_ASSOC                        HCI_OPCODE(0x05,0x00a)
#define HCI_Write_Remote_AMP_ASSOC                      HCI_OPCODE(0x05,0x00b)

// 7.6 TESTING COMMANDS
#define HCI_Read_Loopback_Mode                          HCI_OPCODE(0x06,0x001)
#define HCI_Write_Loopback_Mode                         HCI_OPCODE(0x06,0x002)
#define HCI_Enable_Device_Under_Test_Mode               HCI_OPCODE(0x06,0x003)
#define HCI_Write_Simple_Pairing_Debug_Mode             HCI_OPCODE(0x06,0x004)
#define HCI_Enable_AMP_Receiver_Reports                 HCI_OPCODE(0x06,0x007)
#define HCI_AMP_Test_End                                HCI_OPCODE(0x06,0x008)
#define HCI_AMP_Test                                    HCI_OPCODE(0x06,0x009)

// 7.8 LE CONTROLLER COMMANDS
#define HCI_LE_Set_Event_Mask                           HCI_OPCODE(0x08,0x001)
#define HCI_LE_Read_Buffer_Size                         HCI_OPCODE(0x08,0x002)
#define HCI_LE_Read_Local_Supported_Features            HCI_OPCODE(0x08,0x003)
#define HCI_LE_Set_Random_Address                       HCI_OPCODE(0x08,0x005)
#define HCI_LE_Set_Advertising_Parameters               HCI_OPCODE(0x08,0x006)
#define HCI_LE_Read_Advertising_Channel_TX_Power        HCI_OPCODE(0x08,0x007)
#define HCI_LE_Set_Advertising_Data                     HCI_OPCODE(0x08,0x008)
#define HCI_LE_Set_Scan_Response_Data                   HCI_OPCODE(0x08,0x009)
#define HCI_LE_Set_Advertise_Enable                     HCI_OPCODE(0x08,0x00a)
#define HCI_LE_Set_Scan_Parameters                      HCI_OPCODE(0x08,0x00b)
#define HCI_LE_Set_Scan_Enable                          HCI_OPCODE(0x08,0x00c)
#define HCI_LE_Create_Connection                        HCI_OPCODE(0x08,0x00d)
#define HCI_LE_Create_Connection_Cancel                 HCI_OPCODE(0x08,0x00e)
#define HCI_LE_Read_White_List_Size                     HCI_OPCODE(0x08,0x00f)
#define HCI_LE_Clear_White_List                         HCI_OPCODE(0x08,0x010)
#define HCI_LE_Add_Device_To_White_List                 HCI_OPCODE(0x08,0x011)
#define HCI_LE_Remove_Device_From_White_List            HCI_OPCODE(0x08,0x012)
#define HCI_LE_Connection_Update                        HCI_OPCODE(0x08,0x013)
#define HCI_LE_Set_Host_Channel_Classification          HCI_OPCODE(0x08,0x014)
#define HCI_LE_Read_Channel_Map                         HCI_OPCODE(0x08,0x015)
#define HCI_LE_Read_Remote_Used_Features                HCI_OPCODE(0x08,0x016)
#define HCI_LE_Encrypt                                  HCI_OPCODE(0x08,0x017)
#define HCI_LE_Rand                                     HCI_OPCODE(0x08,0x018)
#define HCI_LE_Start_Encryption                         HCI_OPCODE(0x08,0x019)
#define HCI_LE_Long_Term_Key_Request_Reply              HCI_OPCODE(0x08,0x01a)
#define HCI_LE_Long_Term_Key_Requested_Negative_Reply   HCI_OPCODE(0x08,0x01b)
#define HCI_LE_Read_Supported_States                    HCI_OPCODE(0x08,0x01c)
#define HCI_LE_Receiver_Test                            HCI_OPCODE(0x08,0x01d)
#define HCI_LE_Transmitter_Test                         HCI_OPCODE(0x08,0x01e)
#define HCI_LE_Test_End                                 HCI_OPCODE(0x08,0x01f)


//
//  ACL data
//
struct bt_hci_ACL_Data_Packet {
    uint8_t     Connection_Handle[2];
    uint8_t     Data_Total_Length[2];
//  uint8_t     Data[672];              // DEL 2019.03.09  signaling: ACL-U=48, ACL-U(extendedFlow)=672, LE-U=23
    uint8_t     Data[0];                // ADD 2019.03.09
} __attribute__ ((packed));
#define HCI_ACL_Data_Packet_HeadSize            4


//
//  SCO data
//
struct bt_hci_Synchronous_Data_Packet {
    uint8_t     Connection_Handle[2];
    uint8_t     Data_Total_Length[2];
    uint8_t     Data[64];
} __attribute__ ((packed));
#define HCI_Synchronous_Data_Packet_HeadSize    4

#define HCI_CONNECTION_HANDLE_MASK          0b0000111111111111  //
#define HCI_PACKET_BOUNDARY_FLAG_MASK       0b0011000000000000  // in Connection_Handle, ACL data only
#define HCI_PACKET_BOUNDARY_FLAG_FIRST_NOFL 0b0000000000000000  // in Connection_Handle, ACL data only
#define HCI_PACKET_BOUNDARY_FLAG_CONTINUE   0b0001000000000000  // in Connection_Handle, ACL data only
#define HCI_PACKET_BOUNDARY_FLAG_FIRST      0b0010000000000000  // in Connection_Handle, ACL data only
#define HCI_PACKET_BOUNDARY_FLAG_COMPLETE   0b0011000000000000  // in Connection_Handle, ACL data only
#define HCI_BROADCAST_FLAG_MASK             0b1100000000000000  // in Connection_Handle, ACL data only
#define HCI_BROADCAST_FLAG_P2P              0b0000000000000000  // in Connection_Handle, ACL data only
#define HCI_BROADCAST_FLAG_ACTIVE_SLAVE     0b0100000000000000  // in Connection_Handle, ACL data only
#define HCI_BROADCAST_FLAG_PARKED_SLAVE     0b1000000000000000  // in Connection_Handle, ACL data only


//
//  Event
//
struct bt_hci_Event_Packet {
    uint8_t     Event_Code;
    uint8_t     Parameter_Total_Length;
    union {
        struct {
            uint8_t     Status;
        }           Inquiry_Complete;
        struct {
            uint8_t     Num_Responses;
            uint8_t     BD_ADDR[6];                     // *Num_Responses
            uint8_t     Page_Scan_Repetition_Mode[1];   // *Num_Responses
            uint8_t     Reserved[1];                    // *Num_Responses
            uint8_t     Reserved2[1];                   // *Num_Responses
            uint8_t     Class_of_Device[3];             // *Num_Responses
            uint8_t     Clock_Offset[2];                // *Num_Responses
        }           Inquiry_Result;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     BD_ADDR[6];
            uint8_t     Link_Type;
            uint8_t     Encryption_Mode;
        }           Connection_Complete;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Class_of_Device[3];
            uint8_t     Link_Type;
        }           Connection_Request;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Reason;
        }           Disconnection_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
        }           Authentication_Complete;
        struct {
            uint8_t     Status;
            uint8_t     BD_ADDR[6];
            uint8_t     Remote_Name[248];
        }           Remote_Name_Request_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Encryption_Enable;
        }           Encryption_Change;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
        }           Change_Connection_Link_Key_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Key_Flag;
        }           Master_Link_Key_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     LMP_Features[8];
        }           Read_Remote_Supported_Features_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     LMP_Version;
            uint8_t     Manufacturer_Name[2];
            uint8_t     LMP_Subversion[2];
        }           Read_Remote_Version_Information_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Flags;
            uint8_t     Service_Type;
            uint8_t     Token_Rate[4];
            uint8_t     Peak_Bandwidth[4];
            uint8_t     Latency[4];
            uint8_t     Delay_Variation[4];
        }           QoS_Setup_Complete;
        struct {
            uint8_t     Num_HCI_Command_Packets;
            uint8_t     Command_Opcode[2];
            union {
                struct {
                    uint8_t     Status;
                }           general;
                // 7.1 Link control
                struct {
                    uint8_t     Status;
                }           Inquiry_Cancel;
                struct {
                    uint8_t     Status;
                }           Periodic_Inquiry_Mode;
                struct {
                    uint8_t     Status;
                }           Exit_Periodic_Inquiry_Mode;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Create_Connection_Cancel;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Link_Key_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Link_Key_Request_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           PIN_Code_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           PIN_Code_Request_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Remote_Name_Request_Cancel;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                    uint8_t     LMP_Handle;
                    uint8_t     Reserved[4];
                }           Read_LMP_Handle;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           IO_Capability_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           User_Confirmation_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           User_Confirmation_Request_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           User_Passkey_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           User_Passkey_Request_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Remote_OOB_Data_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           Remote_OOB_Data_Request_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];     // CHG 2016.04.21
                }           IO_Capability_Request_Negative_Reply;
                // 7.2 Link policy
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                    uint8_t     Current_Role;
                }           Role_Discovery;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                    uint8_t     Link_Policy_Settings[2];
                }           Read_Link_Policy_Settings;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                }           Write_Link_Policy_Settings;
                struct {
                    uint8_t     Status;
                    uint8_t     Default_Link_Policy_Settings[2];
                }           Read_Default_Link_Policy_Settings;
                struct {
                    uint8_t     Status;
                }           Write_Default_Link_Policy_Settings;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                }           Sniff_Subrating;

                // 7.3 Controller & baseband
                //  :
                //  :

                // 7.4 Information
                struct {
                    uint8_t     Status;
                    uint8_t     HCI_Version;
                    uint8_t     HCI_Revision[2];
                    uint8_t     LMP_Version;
                    uint8_t     Manufacturer_Name[2];
                    uint8_t     LMP_Subversion[2];
                }           Read_Local_Version_Information;
                struct {
                    uint8_t     Status;
                    uint8_t     Supported_Commands[64];
                }           Read_Local_Supported_Commands;
                struct {
                    uint8_t     Status;
                    uint8_t     LMP_Features[8];
                }           Read_Local_Supported_Features;
                struct {
                    uint8_t     Status;
                    uint8_t     Page_Number;
                    uint8_t     Maximum_Page_Number;
                    uint8_t     Extended_LMP_Features[8];
                }           Read_Local_Extended_Features;
                struct {
                    uint8_t     Status;
                    uint8_t     HC_ACL_Data_Packet_Length[2];
                    uint8_t     HC_Synchronous_Data_Packet_Length;
                    uint8_t     HC_Total_Num_ACL_Data_Packets[2];
                    uint8_t     HC_Total_Num_Synchronous_Data_Packets[2];
                }           Read_Buffer_Size;
                struct {
                    uint8_t     Status;
                    uint8_t     BD_ADDR[6];
                }           Read_BD_ADDR;
                struct {
                    uint8_t     Status;
                    uint8_t     Max_ACL_Data_Packet_Length[2];
                    uint8_t     Data_Block_Length[2];
                    uint8_t     Total_Num_Data_Blocks[2];
                }           Read_Data_Block_Size;
                // 7.5 Status
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     Failed_Contact_Counter[2];
                }           Read_Failed_Contact_Counter;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                }           Reset_Failed_Contact_Counter;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     Link_Quality;
                }           Read_Link_Quality;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     RSSI;
                }           Read_RSSI;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     AFH_Mode;
                    uint8_t     AFH_Channel_Map[10];
                }           Read_AFH_Channel_Map;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     Clock[4];
                    uint8_t     Accuracy[2];
                }           Read_Clock;
                struct {
                    uint8_t     Status;
                    uint8_t     Handle[2];
                    uint8_t     Key_Size;
                }           Read_Encryption_Key_Size;
                // 7.6 Testing
                struct {
                    uint8_t     Status;
                    uint8_t     Loopback_Mode;
                }           Read_Loopback_Mode;
                struct {
                    uint8_t     Status;
                }           Write_Loopback_Mode;
                struct {
                    uint8_t     Status;
                }           Enable_Device_Under_Test_Mode;
                struct {
                    uint8_t     Status;
                }           Write_Simple_Pairing_Debug_Mode;
                // 7.8 LE
                struct {
                    uint8_t     Status;
                }           LE_Set_Event_Mask;
                struct {
                    uint8_t     Status;
                    uint8_t     HC_LE_Data_Packet_Length[2];
                    uint8_t     HC_Total_Num_LE_Data_Packets;
                }           LE_Read_Buffer_Size;
                struct {
                    uint8_t     Status;
                    uint8_t     LE_Features[8];
                }           LE_Read_Local_Supported_Features;
                struct {
                    uint8_t     Status;
                }           LE_Set_Random_Address;
                struct {
                    uint8_t     Status;
                }           LE_Set_Advertising_Parameters;
                struct {
                    uint8_t     Status;
                    uint8_t     Transmit_Power_Level;
                }           LE_Read_Advertising_Channel_Tx_Power;
                struct {
                    uint8_t     Status;
                }           LE_Set_Advertising_Data;
                struct {
                    uint8_t     Status;
                }           LE_Set_Scan_Response_Data;
                struct {
                    uint8_t     Status;
                }           LE_Set_Advertise_Enable;
                struct {
                    uint8_t     Status;
                }           LE_Set_Scan_Parameters;
                struct {
                    uint8_t     Status;
                }           LE_Set_Scan_Enable;
                struct {
                    uint8_t     Status;
                }           LE_Create_Connection_Cancel;
                struct {
                    uint8_t     Status;
                    uint8_t     White_List_Size;
                }           LE_Read_White_List_Size;
                struct {
                    uint8_t     Status;
                }           LE_Clear_White_List;
                struct {
                    uint8_t     Status;
                }           LE_Add_Device_To_White_List;
                struct {
                    uint8_t     Status;
                }           LE_Remove_Device_From_White_List;
                struct {
                    uint8_t     Status;
                }           LE_Set_Host_Channel_Classification;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                    uint8_t     Channel_Map[5];
                }           LE_Read_Channel_Map;
                struct {
                    uint8_t     Status;
                }           LE_Encrypt;
                struct {
                    uint8_t     Status;
                    uint8_t     Random_Number[8];
                }           LE_Rand;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                }           LE_Long_Term_Key_Request_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     Connection_Handle[2];
                }           LE_Long_Term_Key_Requested_Negative_Reply;
                struct {
                    uint8_t     Status;
                    uint8_t     LE_States[8];
                }           LE_Read_Supported_States;
                struct {
                    uint8_t     Status;
                }           LE_Receiver_Test;
                struct {
                    uint8_t     Status;
                }           LE_Transmitter_Test;
                struct {
                    uint8_t     Status;
                    uint8_t     Number_Of_Packets[2];
                }           LE_Test_End;

            }           Return_Parameter;        // Option
        }           Command_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Num_HCI_Command_Packets;
            uint8_t     Command_Opcode[2];
        }           Command_Status;
        struct {
            uint8_t     Hardware_Code;
        }           Hardware_Error;
                                        // ADD START 2015.02.06
        struct {
            uint8_t     Handle[2];
        }           Flush_Occurred;
        struct {
            uint8_t     Status;
            uint8_t     BD_ADDR[6];
            uint8_t     New_Role;
        }           Role_Change;
                                        // ADD END 2015.02.06
        struct {
            uint8_t     Number_of_Handles;
            uint8_t     Connection_Handle[2];           // *Number_of_Handles
            uint8_t     HC_Num_Of_Completed_Packets[2]; // *Number_of_Handles
        }           Number_Of_Completed_Packets;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Current_Mode;
            uint8_t     Interval[2];
        }           Mode_Change;
        struct {
            uint8_t     Num_Keys;
            uint8_t     BD_ADDR[6];
            uint8_t     Link_Key[16];
        }           Return_Link_Keys;
        struct {
            uint8_t     BD_ADDR[6];
        }           PIN_Code_Request;
        struct {
            uint8_t     BD_ADDR[6];
        }           Link_Key_Request;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Link_Key[16];
            uint8_t     Key_Type;
        }           Link_Key_Notification;
                                        // ADD START 2015.02.06
        struct {
            uint8_t     HCI_Command_Packet[1];
        }           Loopback_Command;
                                        // ADD END 2015.02.06
        struct {
            uint8_t     Link_Type;
        }           Data_Buffer_Overflow;
        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     LMP_Max_Slots;
        }           Max_Slots_Change;
                                        // ADD START 2015.02.06
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Clock_Offset[2];
        }           Read_Clock_Offset_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Packet_Type[2];
        }           Connection_Packet_Type_Changed;
        struct {
            uint8_t     Handle[2];
        }           QoS_Violation;
                                        // ADD END 2015.02.06
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Page_Scan_Repetition_Mode;
        }           Page_Scan_Repetition_Mode_Change;
                                        // ADD START 2015.02.06
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Flags;
            uint8_t     Flow_direction;
            uint8_t     Token_Rate[4];
            uint8_t     Peak_Bandwidth[4];
            uint8_t     Access_Latency[4];
        }           Flow_Specification_Complete;
        struct {
            uint8_t     Number_of_Handles;
            uint8_t     BD_ADDR[6];                     // *Number_of_Handles
            uint8_t     Page_Scan_Repetition_Mode[1];   // *Number_of_Handles
            uint8_t     Reserved[1];                    // *Number_of_Handles
            uint8_t     Class_of_Device[3];             // *Number_of_Handles
            uint8_t     Clock_Offset[2];                // *Number_of_Handles
            uint8_t     RSSI[1];                        // *Number_of_Handles
        }           Inquiry_Result_with_RSSI;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Page_Number;
            uint8_t     Maximum_Page_Number;
            uint8_t     Extended_LMP_Features[8];
        }           Read_Remote_Extended_Features_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     BD_ADDR[6];
            uint8_t     Link_Type;
            uint8_t     Transmission_Interval;
            uint8_t     Retransmission_window;
            uint8_t     Rx_Packet_Length[2];
            uint8_t     Tx_Packet_Length[2];
            uint8_t     Air_Mode;
        }           Synchronous_Connection_Complete;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Transmission_Interval;
            uint8_t     Retransmission_window;
            uint8_t     Rx_Packet_Length[2];
            uint8_t     Tx_Packet_Length[2];
        }           Synchronous_Connection_Changed;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Maximum_Transmit_Latency[2];
            uint8_t     Maximum_Receive_Latency[2];
            uint8_t     Minimum_Remote_Timeout[2];
            uint8_t     Minimum_Local_Timeout[2];
        }           Sniff_Subrating;
        struct {
            uint8_t     Num_Responses;
            uint8_t     BD_ADDR[6];                         // *Num_Responses
            uint8_t     Page_Scan_Repetition_Mode[1];       // *Num_Responses
            uint8_t     Reserved[1];                        // *Num_Responses
            uint8_t     Class_of_Device[3];                 // *Num_Responses
            uint8_t     Clock_Offset[2];                    // *Num_Responses
            uint8_t     RSSI[1];                            // *Num_Responses
//          uint8_t     RExtended_Inquiry_Response[240];    // *Num_Responses  DEL 2019.03.09
            uint8_t     Extended_Inquiry_Response[240];     // *Num_Responses  ADD 2019.03.09
        }           Extended_Inquiry_Result;
        struct {
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
        }           Encryption_Key_Refresh_Complete;
        struct {
            uint8_t     BD_ADDR[6];
        }           IO_Capability_Request;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     IO_Capability;
            uint8_t     OOB_Data_Present;
            uint8_t     Authentication_Requirements;
        }           IO_Capability_Response;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Numeric_Value[4];      // 0..999999
        }           User_Confirmation_Request;
        struct {
            uint8_t     BD_ADDR[6];
        }           User_Passkey_Request;
        struct {
            uint8_t     BD_ADDR[6];
        }           Remote_OOB_Data_Request;
        struct {
            uint8_t     Status;
            uint8_t     BD_ADDR[6];
        }           Simple_Pairing_Complete;
        struct {
            uint8_t     Connection_Handle[2];
            uint8_t     Link_Supervision_Timeout[2];
        }           Link_Supervision_Timeout_Changed;
        struct {
            uint8_t     Connection_Handle[2];
        }           Enhanced_Flush_Complete;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Passkey[4];     // 0..999999
        }           User_Passkey_Notification;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Notification_Type;
        }           Keypress_Notification;
        struct {
            uint8_t     BD_ADDR[6];
            uint8_t     Host_Supported_Features[8];
        }           Remote_Host_Supported_Features_Notification;
                                        // ADD END 2015.02.06
        struct {
            uint8_t     Subevent_Code;
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Role;
            uint8_t     Peer_Address_Type;
            uint8_t     Peer_Address[6];
            uint8_t     Conn_Interval[2];
            uint8_t     Conn_Latency[2];
            uint8_t     Supervision_Timeout[2];
            uint8_t     Master_Clock_Accuracy;
        }           LE_Connection_Complete;
        struct {
            uint8_t     Subevent_Code;
            uint8_t     Num_Reports;
            uint8_t     Event_Type[1];      // *Num_Reports
            uint8_t     Address_Type[1];    // *Num_Reports
            uint8_t     Address[6];         // *Num_Reports
            uint8_t     Length_Data[1];     // *Num_Reports
            uint8_t     Data[1];            // SUM(Length_Data)
            int8_t      RSSI[1];            // *Num_Reports
        }           LE_Advertising_Report;
        struct {
            uint8_t     Subevent_Code;
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     Conn_Interval[2];
            uint8_t     Conn_Latency[2];
            uint8_t     Supervision_Timeout[2];
        }           LE_Connection_Update_Complete;
        struct {
            uint8_t     Subevent_Code;
            uint8_t     Status;
            uint8_t     Connection_Handle[2];
            uint8_t     LE_Features[8];
        }           LE_Read_Remote_Used_Features_Complete;
        struct {
            uint8_t     Subevent_Code;
            uint8_t     Connection_Handle[2];
            uint8_t     Random_Number[8];
            uint8_t     Encryption_Diversifier[2];
        }           LE_Long_Term_Key_Request;

    }            Event_Parameter;
} __attribute__ ((packed));

#define HCI_Event_Packet_HeadSize    2

#define HCI_Event_Inquiry_Complete                          0x01
#define HCI_Event_Inquiry_Result                            0x02
#define HCI_Event_Connection_Complete                       0x03
#define HCI_Event_Connection_Request                        0x04
#define HCI_Event_Disconnection_Complete                    0x05
#define HCI_Event_Authentication_Complete                   0x06
#define HCI_Event_Remote_Name_Request_Complete              0x07
#define HCI_Event_Encryption_Change                         0x08
#define HCI_Event_Change_Connection_Link_Key_Complete       0x09
#define HCI_Event_Master_Link_Key_Complete                  0x0a
#define HCI_Event_Read_Remote_Supported_Features_Complete   0x0b
#define HCI_Event_Read_Remote_Version_Information_Complete  0x0c
#define HCI_Event_QoS_Setup_Complete                        0x0d
#define HCI_Event_Command_Complete                          0x0e
#define HCI_Event_Command_Status                            0x0f
#define HCI_Event_Hardware_Error                            0x10
#define HCI_Event_Flush_Occurred                            0x11
#define HCI_Event_Role_Change                               0x12
#define HCI_Event_Number_Of_Completed_Packets               0x13
#define HCI_Event_Mode_Change                               0x14
#define HCI_Event_Return_Link_Keys                          0x15
#define HCI_Event_PIN_Code_Request                          0x16
#define HCI_Event_Link_Key_Request                          0x17
#define HCI_Event_Link_Key_Notification                     0x18
#define HCI_Event_Data_Buffer_Overflow                      0x1a
#define HCI_Event_Max_Slots_Change                          0x1b
#define HCI_Event_Read_Clock_Offset_Complete                0x1c
#define HCI_Event_Connection_Packet_Type_Changed            0x1d
#define HCI_Event_QoS_Violation                             0x1e
#define HCI_Event_Page_Scan_Repetition_Mode_Change          0x20
#define HCI_Event_Flow_Specification_Complete               0x21
#define HCI_Event_Inquiry_Result_with_RSSI                  0x22
#define HCI_Event_Read_Remote_Extended_Features_Complete    0x23
#define HCI_Event_Synchronous_Connection_Complete           0x2c
#define HCI_Event_Synchronous_Connection_Changed            0x2d
#define HCI_Event_Sniff_Subrating                           0x2e
#define HCI_Event_Extended_Inquiry_Result                   0x2f
#define HCI_Event_Encryption_Key_Refresh_Complete           0x30
#define HCI_Event_IO_Capability_Request                     0x31
#define HCI_Event_IO_Capability_Response                    0x32
#define HCI_Event_User_Confirmation_Request                 0x33
#define HCI_Event_User_Passkey_Request                      0x34
#define HCI_Event_Remote_OOB_Data_Request                   0x35
#define HCI_Event_Simple_Pairing_Complete                   0x36
#define HCI_Event_Link_Supervision_Timeout_Changed          0x38
#define HCI_Event_Enhanced_Flush_Complete                   0x39
#define HCI_Event_User_Passkey_Notification                 0x3b
#define HCI_Event_Keypress_Notification                     0x3c
#define HCI_Event_Remote_Host_Supported_Features_Notification   0x3d
#define HCI_Event_Physical_Link_Complete                    0x40
#define HCI_Event_Channel_Selected                          0x41
#define HCI_Event_Disconnection_Physical_Link_Complete      0x42
#define HCI_Event_Physical_Link_Loss_Early_Warning          0x43
#define HCI_Event_Physical_Link_Recovery                    0x44
#define HCI_Event_Logical_Link_Complete                     0x45
#define HCI_Event_Disconnection_Logical_Link_Complete       0x46
#define HCI_Event_Flow_Spec_Modify_Complete                 0x47
#define HCI_Event_Number_Of_Completed_Data_Blocks           0x48
#define HCI_Event_Short_Range_Mode_Change_Complete          0x4c
#define HCI_Event_AMP_Status_Change                         0x4d
#define HCI_Event_AMP_Start_Test                            0x49
#define HCI_Event_AMP_Test_End                              0x4a
#define HCI_Event_AMP_Receiver_Report                       0x4b

#define HCI_Event_LE_Meta_Event                             0x3e
#define HCI_Event_LE_Connection_Complete                    0x01
#define HCI_Event_LE_Advertising_Report                     0x02
#define HCI_Event_LE_Connection_Update_Complete             0x03
#define HCI_Event_LE_Read_Remote_Used_Features_Complete     0x04
#define HCI_Event_LE_Long_Term_Key_Request                  0x05

//
//  Error code
//
#define HCI_Error_None                                          0x00
#define HCI_Error_Unknown_HCI_Command                           0x01
#define HCI_Error_Unknown_Connection_Identifier                 0x02
#define HCI_Error_Hardware_Failure                              0x03
#define HCI_Error_Page_Timeout                                  0x04
#define HCI_Error_Authentication_Failure                        0x05
#define HCI_Error_PIN_or_key_Missing                            0x06
#define HCI_Error_Memory_Capacity_Exceeded                      0x07
#define HCI_Error_Connection_Timeout                            0x08
#define HCI_Error_Connection_Limit_Exceeded                     0x09
#define HCI_Error_Synchronous_Connection_Limit                  0X0a
#define HCI_Error_ACL_Connection_Already_Exists                 0x0b
#define HCI_Error_Command_Disallowed                            0x0c
#define HCI_Error_Connection_Rejected_Limited_Resources         0x0d
#define HCI_Error_Connection_Rejected_Security_Reasons          0x0e
#define HCI_Error_Connection_Rejected_Unacceptable_BD_ADDR      0x0f
#define HCI_Error_Connection_Accept_Timeout_Exceeded            0x10
#define HCI_Error_Unsupported_Feature_or_Parameter_Value        0x11
#define HCI_Error_Invalid_HCI_Command_Parameters                0x12
#define HCI_Error_Remote_User_Terminated_Connection             0x13
#define HCI_Error_Remote_Device_Terminated_Connection_Resource  0x14
#define HCI_Error_Remote_Device_Terminated_Connection_PowerOff  0x15
#define HCI_Error_Connection_Terminated_by_Local_Host           0x16
#define HCI_Error_Repeated_Attempts                             0x17
#define HCI_Error_Pairing_not_Allowed                           0x18
#define HCI_Error_Unknown_LMP_PDU                               0x19
#define HCI_Error_Unsupported_Remote_LMP_Feature                0x1a
#define HCI_Error_SCO_Offset_Rejected                           0x1b
#define HCI_Error_SCO_Interval_Rejected                         0x1c
#define HCI_Error_SCO_Air_Mode_Rejected                         0x1d
#define HCI_Error_Invalid_LMP_Parameters                        0x1e
#define HCI_Error_Unspecified_Error                             0x1f
#define HCI_Error_Unsupported_LMP_Parameter_Value               0x20
#define HCI_Error_Role_Change_Not_Allowed                       0x21
#define HCI_Error_LMP_Response_Timeout                          0x22
#define HCI_Error_LMP_Error_Transaction_Collision               0x23
#define HCI_Error_LMP_PDU_Not_Allowed                           0x24
#define HCI_Error_Encryption_Mode_Not_Acceptable                0x25
#define HCI_Error_Link_Key_Can_Not_be_Changed                   0x26
#define HCI_Error_Requested_Qos_Not_Supported                   0x27
#define HCI_Error_Instant_Passed                                0x28
#define HCI_Error_Pairing_with_Unit_Key_Not_Supported           0x29
#define HCI_Error_Different_Transaction_Collision               0x2a
#define HCI_Error_QoS_Unacceptable_Parameter                    0x2c
#define HCI_Error_QoS_Rejected                                  0x2d
#define HCI_Error_Channel_Assessment_Not_Supported              0x2e
#define HCI_Error_Insufficient_Security                         0x2f
#define HCI_Error_Parameter_out_of_Mandatory_Range              0x30
#define HCI_Error_Role_Switch_Pending                           0x32
#define HCI_Error_Reserved_Slot_Violation                       0x34
#define HCI_Error_Role_Switch_Failed                            0x35
#define HCI_Error_Extended_Inquiry_Response_Too_Large           0x36
#define HCI_Error_Simple_Pairing_Not_Supported_By_Host          0x37
#define HCI_Error_Host_Busy_Pairing                             0x38
#define HCI_Error_Connection_Rejected_Due_To_No_Suitable_Channel_Found  0x39
#define HCI_Error_Controller_Busy                               0x3a
#define HCI_Error_Unacceptable_Connection_Interval              0x3b
#define HCI_Error_Directed_Advertising_Timeout                  0x3c
#define HCI_Error_Connection_Terminated_Due_To_MIC_Failure      0x3d
#define HCI_Error_Connection_Failed_To_Be_Established           0x3e
#define HCI_Error_MAC_Connection_Failed                         0x3f

//
//  Parameter
//
#define HCI_Parm_Scan_Enable_Inquiry_Scan                       0x01
#define HCI_Parm_Scan_Enable_Page_Scan                          0x02
#define HCI_Parm_Inquiry_Scan_Type_Mandatory                    0x00
#define HCI_Parm_Inquiry_Mode_Standard                          0x00
#define HCI_Parm_Inquiry_Mode_RSSI                              0x01
#define HCI_Parm_Page_Scan_Period_Mode_P0                       0x00
#define HCI_Parm_Page_Scan_Period_Mode_P1                       0x01
#define HCI_Parm_Page_Scan_Period_Mode_P2                       0x02
#define HCI_Parm_Page_Scan_Type_Mandatory                       0x00
#define HCI_Parm_Page_Scan_Type_Optional                        0x01
#define HCI_Parm_PIN_Type_Variable                              0x00
#define HCI_Parm_PIN_Type_Fixed                                 0x01
#define HCI_Parm_Authentication_Enable_not_required             0x00
#define HCI_Parm_Authentication_Enable_required                 0x01
#define HCI_Parm_Encryption_Mode_not_required                   0x00
#define HCI_Parm_Encryption_Mode_required                       0x01
#define HCI_Parm_Hold_Mode_Activity_Power_State                 0x00
#define HCI_Parm_Hold_Mode_Activity_Suspend_Page_Scan           0x01
#define HCI_Parm_Hold_Mode_Activity_Suspend_Inquiry_Scan        0x02
#define HCI_Parm_Hold_Mode_Activity_Suspend_Periodic_Inquiries  0x04
#define HCI_Parm_Link_Policy_Settings_Disable_all               0x0000
#define HCI_Parm_Link_Policy_Settings_Role_Switch               0x0001
#define HCI_Parm_Link_Policy_Settings_Hold_Mode                 0x0002
#define HCI_Parm_Link_Policy_Settings_Sniff_Mode                0x0004
#define HCI_Parm_Link_Policy_Settings_Park_State                0x0008
#define HCI_Parm_Synchronous_Flow_Control_Disabled              0x00
#define HCI_Parm_Synchronous_Flow_Control_Enabled               0x01
#define HCI_Parm_Flow_Control_Disable                           0x00		// ADD 2019.03.09
#define HCI_Parm_Flow_Control_Enable_ACL                        0x01		// ADD 2019.03.09
#define HCI_Parm_Flow_Control_Enable_Synchronous                0x02		// ADD 2019.03.09
#define HCI_Parm_Flow_Control_Enable_ACL_Synchronous            0x03		// ADD 2019.03.09
#define HCI_Parm_Packet_Type_DM1                                0x0008
#define HCI_Parm_Packet_Type_DH1                                0x0010
#define HCI_Parm_Packet_Type_DM3                                0x0400
#define HCI_Parm_Packet_Type_DH3                                0x0800
#define HCI_Parm_Packet_Type_DM5                                0x4000
#define HCI_Parm_Packet_Type_DH5                                0x8000
#define HCI_Parm_Packet_Type_2_DH1_Not                          0x0002
#define HCI_Parm_Packet_Type_3_DH1_Not                          0x0004
#define HCI_Parm_Packet_Type_2_DH3_Not                          0x0100
#define HCI_Parm_Packet_Type_3_DH3_Not                          0x0200
#define HCI_Parm_Packet_Type_2_DH5_Not                          0x1000
#define HCI_Parm_Packet_Type_3_DH5_Not                          0x2000
#define HCI_Parm_Page_Scan_Repetition_Mode_R0                   0x00
#define HCI_Parm_Page_Scan_Repetition_Mode_R1                   0x01
#define HCI_Parm_Page_Scan_Repetition_Mode_R2                   0x02
#define HCI_Parm_Allow_Role_Switch_NotAccept                    0x00
#define HCI_Parm_Allow_Role_Switch_Accept                       0x01
#define HCI_Parm_Link_Type_SCO                                  0x00
#define HCI_Parm_Link_Type_ACL                                  0x01
#define HCI_Parm_Link_Type_eSCO                                 0x02
#define HCI_Parm_Encryption_Enable_Off                          0x00
#define HCI_Parm_Encryption_Enable_On                           0x01
#define HCI_Parm_Key_Flag_Using_Semi_Permanent                  0x00
#define HCI_Parm_Key_Flag_Using_Temporary                       0x01
#define HCI_Parm_Service_Type_No_Traffic                        0x00
#define HCI_Parm_Service_Type_Best_Effort                       0x01
#define HCI_Parm_Service_Type_Guaranteed                        0x02
#define HCI_Parm_Role_Become_the_Master                         0x00
#define HCI_Parm_Role_Remain_the_Slave                          0x01
#define HCI_Parm_Key_Type_Combination_Key                       0x00
#define HCI_Parm_Key_Type_Local_Unit_Key                        0x01
#define HCI_Parm_Key_Type_Remote_Unit_Key                       0x02
#define HCI_Parm_Key_Type_Debug_Combination_Key                 0x03
#define HCI_Parm_Key_Type_Unauthenticated_Combination_Key       0x04
#define HCI_Parm_Key_Type_Authenticated_Combination_Key         0x05
#define HCI_Parm_Key_Type_Changed_Combination_Key               0x06
#define HCI_Parm_HCI_Version_1_0b                               0
#define HCI_Parm_HCI_Version_1_1                                1
#define HCI_Parm_HCI_Version_1_2                                2
#define HCI_Parm_HCI_Version_2_0_EDR                            3
#define HCI_Parm_HCI_Version_2_1_EDR                            4
#define HCI_Parm_HCI_Version_3_0_HS                             5
#define HCI_Parm_HCI_Version_4_0                                6
#define HCI_Parm_Role_Master                                    0x00
#define HCI_Parm_Role_Slave                                     0x01
#define HCI_Parm_Role_Unknown                                   0xff
#define HCI_Parm_Which_Clock_Local                              0x00
#define HCI_Parm_Which_Clock_Piconet                            0x01
#define HCI_Parm_Loopback_Mode_Disable                          0x00
#define HCI_Parm_Loopback_Mode_Enable_Local                     0x01
#define HCI_Parm_Loopback_Mode_Enable_Remote                    0x02
#define HCI_Parm_Simple_Pairing_Debug_Mode_Disable              0x00
#define HCI_Parm_Simple_Pairing_Debug_Mode_Enable               0x01

#define HCI_Parm_Filter_Type_Clear_All_Filters                  0x00
#define HCI_Parm_Filter_Type_Inquiry_Result                     0x01
#define HCI_Parm_Filter_Type_Connection_Setup                   0x02
#define HCI_Parm_Inquiry_Result_Filter_Condition_Type_All_Device        0x00
#define HCI_Parm_Inquiry_Result_Filter_Condition_Type_Class_of_Device   0x01
#define HCI_Parm_Inquiry_Result_Filter_Condition_Type_BD_ADDR           0x02
#define HCI_Parm_Connection_Setup_Filter_Condition_Type_All_Device      0x00
#define HCI_Parm_Connection_Setup_Filter_Condition_Type_Class_of_Device 0x01
#define HCI_Parm_Connection_Setup_Filter_Condition_Type_BD_ADDR         0x02
#define HCI_Parm_Read_All_Flag_BD_ADDR                          0x00
#define HCI_Parm_Read_All_Flag_All                              0x01
#define HCI_Parm_Delete_All_Flag_BD_ADDR                        0x00
#define HCI_Parm_Delete_All_Flag_All                            0x01
#define HCI_Parm_Authentication_Enable_On                       0x00
#define HCI_Parm_Authentication_Enable_Off                      0x01
#define HCI_Parm_Simple_Pairing_Mode_Disabled                   0x00
#define HCI_Parm_Simple_Pairing_Mode_Enable                     0x01
#define HCI_Parm_Flow_direction_Outgoing                        0x00
#define HCI_Parm_Flow_direction_Incoming                        0x01
#define HCI_Parm_Air_Mode_u_law_log                             0x00
#define HCI_Parm_Air_Mode_A_law_log                             0x01
#define HCI_Parm_Air_Mode_CVSD                                  0x02
#define HCI_Parm_Air_Mode_Transparent_Data                      0x03
#define HCI_Parm_IO_Capability_DisplayOnly                      0x00
#define HCI_Parm_IO_Capability_DisplayYesNo                     0x01
#define HCI_Parm_IO_Capability_KeyboardOnly                     0x02
#define HCI_Parm_IO_Capability_NoInputNoOutput                  0x03
#define HCI_Parm_OOB_Data_Present_Not                           0x00
#define HCI_Parm_OOB_Data_Present_From_Remote_Device            0x01
#define HCI_Parm_Authentication_Requirements_Numeric_Comparison                     0x00
#define HCI_Parm_Authentication_Requirements_Use_IO_Capabilities                    0x01
#define HCI_Parm_Authentication_Requirements_Dedicated_Bonding_Numeric_Comparison   0x02
#define HCI_Parm_Authentication_Requirements_Dedicated_Bonding_Use_IO_Capabilities  0x03
#define HCI_Parm_Authentication_Requirements_General_Bonding_Numeric_Comparison     0x04
#define HCI_Parm_Authentication_Requirements_General_Bonding_Use_IO_capabilities    0x05
#define HCI_Parm_Notification_Type_Passkey_entry_started        0
#define HCI_Parm_Notification_Type_Passkey_digit_entered        1
#define HCI_Parm_Notification_Type_Passkey_digit_erased         2
#define HCI_Parm_Notification_Type_Passkey_cleared              3
#define HCI_Parm_Notification_Type_Passkey_entry_completed      4

#define HCI_Parm_LE_Event_Mask_LE_Connection_Complete                   0x01
#define HCI_Parm_LE_Event_Mask_LE_Advertising_Report                    0x02
#define HCI_Parm_LE_Event_Mask_LE_Connection_Update_Complete            0x04
#define HCI_Parm_LE_Event_Mask_LE_Read_Remote_Used_Features_Complete    0x08
#define HCI_Parm_LE_Event_Mask_LE_Long_Term_Key_Request                 0x10
#define HCI_Parm_LE_Event_Mask_All                                      0x1f
#define HCI_Parm_Advertising_Type_ADV_IND                               0x00
#define HCI_Parm_Advertising_Type_ADV_DIRECT_IND                        0x01
#define HCI_Parm_Advertising_Type_ADV_SCAN_IND                          0x02
#define HCI_Parm_Advertising_Type_ADV_NONCONN_IND                       0x03
#define HCI_Parm_Address_Type_Public_Device_Address                     0x00
#define HCI_Parm_Address_Type_Random_Device_Address                     0x01
#define HCI_Parm_Advertising_Channel_Map_37                             0x01
#define HCI_Parm_Advertising_Channel_Map_38                             0x02
#define HCI_Parm_Advertising_Channel_Map_39                             0x04
#define HCI_Parm_Advertising_Channel_Map_Default                        0x07
#define HCI_Parm_Advertising_Filter_Policy_Any                          0x00
#define HCI_Parm_Advertising_Filter_Policy_Scan_restricted              0x01
#define HCI_Parm_Advertising_Filter_Policy_Connect_restricted           0x02
#define HCI_Parm_Advertising_Filter_Policy_Scan_Connect_restricted      0x03
#define HCI_Parm_Advertising_Disabled                                   0x00
#define HCI_Parm_Advertising_Enabled                                    0x01
#define HCI_Parm_Address_Type_Public                                    0x00    // ADD 2017.01.18  Peer_Address_Type
#define HCI_Parm_Address_Type_Random                                    0x01    // ADD 2017.01.18  Peer_Address_Type


//
//  Implement tool
//
// Byte order normalize
#ifdef DELETE                           // DEL START 2015.02.11
#define HCI_UINT16(a)       (uint16_t)(a[0] | (a[1] << 8))
#define HCI_UINT24(a)       (uint32_t)(a[0] | (a[1] << 8) | (a[2] << 16))
#define HCI_UINT32(a)       (uint32_t)(a[0] | (a[1] << 8) | (a[2] << 16) | (a[3] << 24))
#define HCI_STORE16(d,s)    (d[0] = (uint8_t)(s), d[1] = (uint8_t)((s) >> 8))
#define HCI_STORE24(d,s)    (d[0] = (uint8_t)(s), d[1] = (uint8_t)((s) >> 8), d[2] = (uint8_t)((s) >> 16))
#define HCI_STORE32(d,s)    (d[0] = (uint8_t)(s), d[1] = (uint8_t)((s) >> 8), d[2] = (uint8_t)((s) >> 16), d[3] = (uint8_t)((s) >> 24))
#else                                   // DEL END / ADD START 2015.02.11
#define HCI_UINT16(a)       (uint16_t)((a)[0] | ((a)[1] << 8))
#define HCI_UINT24(a)       (uint32_t)((a)[0] | ((a)[1] << 8) | ((a)[2] << 16))
#define HCI_UINT32(a)       (uint32_t)((a)[0] | ((a)[1] << 8) | ((a)[2] << 16) | ((a)[3] << 24))
#define HCI_STORE16(d,s)    ((d)[0] = (uint8_t)(s), (d)[1] = (uint8_t)((s) >> 8))
#define HCI_STORE24(d,s)    ((d)[0] = (uint8_t)(s), (d)[1] = (uint8_t)((s) >> 8), (d)[2] = (uint8_t)((s) >> 16))
#define HCI_STORE32(d,s)    ((d)[0] = (uint8_t)(s), (d)[1] = (uint8_t)((s) >> 8), (d)[2] = (uint8_t)((s) >> 16), (d)[3] = (uint8_t)((s) >> 24))
#define HCI_OPCODE_DATA(a)  (unsigned char)(a), (unsigned char)((a) >> 8)
#define HCI_UINT16_DATA(a)  (unsigned char)(a), (unsigned char)((a) >> 8)
#define HCI_UINT24_DATA(a)  (unsigned char)(a), (unsigned char)((a) >> 8), (unsigned char)((a) >> 16)
#define HCI_UINT32_DATA(a)  (unsigned char)(a), (unsigned char)((a) >> 8), (unsigned char)((a) >> 16), (unsigned char)((a) >> 24)
#endif                                  // ADD END 2015.02.11
                                        // ADD START 2015.02.20
#define HCI_UUID128_DATA(aaaaaaaa,bbbb,cccc,dddd,eeeeeeeeeeee) \
    (uint8_t)(eeeeeeeeeeee),\
    (uint8_t)((eeeeeeeeeeee)>>8),\
    (uint8_t)((eeeeeeeeeeee)>>16),\
    (uint8_t)((eeeeeeeeeeee)>>24),\
    (uint8_t)((eeeeeeeeeeee)>>32),\
    (uint8_t)((eeeeeeeeeeee)>>40),\
    (uint8_t)(dddd),\
    (uint8_t)((dddd)>>8),\
    (uint8_t)(cccc),\
    (uint8_t)((cccc)>>8),\
    (uint8_t)(bbbb),\
    (uint8_t)((bbbb)>>8),\
    (uint8_t)(aaaaaaaa),\
    (uint8_t)((aaaaaaaa)>>8),\
    (uint8_t)((aaaaaaaa)>>16),\
    (uint8_t)((aaaaaaaa)>>24)
                                        // ADD END 2015.02.20

#define HCI_LAP_GENERAL_INQUIRY             0x9E8B33


#endif /* BT_SPEC_HCI_H_ */
