//
//  bt_sdp_server.c
//
//  Bluetooth Protocol Stack - SDP Specification definition
//  Copyright (C) 2015-2018 Toyohiko Togashi tog001@nifty.com
//
//
//  This program is free software; you can redistribute it and/or modify it under the terms of the
//  GNU General Public License as published by the Free Software Foundation; either version 3
//  of the License, or (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
//  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  See the GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along with this program.
//  If not, see <http://www.gnu.org/licenses/>
//
//
//  Reference:
//      Bluetooth SIG (www.bluetooth.com)
//          BLUETOOTH SPECIFICATION Version 2.0 + EDR
//          Volume 3: Core System Package [Host volume]
//          PartB: Service Discovery Protocol (SDP) Specification
//
//  Update history
//  ---------- ----- -------------------------------------------
//  2015.02.06 v0.3a Separated from bt_sdp_server.c
//  2017.03.10 v0.4  Supported an AVDTP
//  2018.02.10 v0.4a Bug fix (ServiceSearchAttributeResponse item names)
//
#ifndef BT_SPEC_SDP_H
#define	BT_SPEC_SDP_H

#include <stdint.h>                     // ADD 2017.03.10

//
//  Protocol format
//
struct bt_sdpFormat {
    uint8_t PDU_ID;
    uint8_t TransactionID[2];
    uint8_t ParameterLength[2];
    union {
        struct {
            uint8_t ErrorCode[2];
        }       ErrorResponse;
        struct {
            uint8_t ServiceSearchPattern[1];        // Data Element Sequence UUID format
//          uint8_t MaximumServiceRecordCount[2];
//          uint8_t ContinuationState[1..17];
//          uint8_t ServiceSearchPattern[];
        }       ServiceSearchRequest;
        struct {
            uint8_t TotalServiceRecordCount[2];
            uint8_t CurrentServiceRecordCount[2];
            uint8_t ServiceRecordHandleList[];      // CurrentServiceRecordCount*4 bytes
//          uint8_t ContinuationState[1..17];
        }       ServiceSearchResponse;
        struct {
            uint8_t ServiceRecordHandle[4];
            uint8_t MaximumAttributeByteCount[2];
            uint8_t AttributeIDList[];              // Data Element Sequence
//          uint8_t ContinuationState[1..17];
        }       ServiceAttributeRequest;
        struct {
            uint8_t AttributeListByteCount[2];
            uint8_t AttributeList[];
//          uint8_t ContinuationState[1..17];
        }       ServiceAttributeResponse;
        struct {
            uint8_t ServiceSearchPattern[5];        // Data Element Sequence UUID format
//          uint8_t ServiceSearchPattern[1];        // Data Element Sequence UUID format
//          uint8_t MaximumAttributeByteCount[2];
//          uint8_t AttributeIDList[];              // Data Element Sequence
//          uint8_t ContinuationState[1..17];
        }       ServiceSearchAttributeRequest;
        struct {
//          uint8_t AttributeListByteCount[2];      // DEL 2018.02.10
//          uint8_t AttributeList[];                // DEL 2018.02.10
            uint8_t AttributeListsByteCount[2];     // ADD 2018.02.10
            uint8_t AttributeLists[];               // ADD 2018.02.10
//          uint8_t ContinuationState[1..17];
        }       ServiceSearchAttributeResponse;
    }       Parameter;
}   __attribute__ ((packed));

#define SDP_HEAD_SIZE                               5

#define SDP_ErrorResponse                           0x01
#define SDP_ServiceSearchRequest                    0x02
#define SDP_ServiceSearchResponse                   0x03
#define SDP_ServiceAttributeRequest                 0x04
#define SDP_ServiceAttributeResponse                0x05
#define SDP_ServiceSearchAttributeRequest           0x06
#define SDP_ServiceSearchAttributeResponse          0x07

#define SDP_ErrorCode_Invalid_unsupported_SDP_version           0x0001
#define SDP_ErrorCode_Invalid_Service_Record_Handle             0x0002
#define SDP_ErrorCode_Invalid_request_syntax                    0x0003
#define SDP_ErrorCode_Invalid_PDU_Size                          0x0004
#define SDP_ErrorCode_Invalid_Continuation_State                0x0005
#define SDP_ErrorCode_Insufficient_Resources_to_satisfy_Request 0x0006

#define SDP_UINT16(a)       (uint16_t)(                                  ((a)[0] << 8) | (a)[1])
#define SDP_UINT24(a)       (uint32_t)(                 ((a)[0] << 16) | ((a)[1] << 8) | (a)[2])
#define SDP_UINT32(a)       (uint32_t)(((a)[0] << 24) | ((a)[1] << 16) | ((a)[2] << 8) | (a)[3])
#define SDP_STORE16(d,s)    (                                                              (d)[0] = (uint8_t)((s) >> 8), (d)[1] = (uint8_t)(s))
#define SDP_STORE24(d,s)    (                               (d)[0] = (uint8_t)((s) >> 16), (d)[1] = (uint8_t)((s) >> 8), (d)[2] = (uint8_t)(s))
#define SDP_STORE32(d,s)    ((d)[0] = (uint8_t)((s) >> 24), (d)[1] = (uint8_t)((s) >> 16), (d)[2] = (uint8_t)((s) >> 8), (d)[3] = (uint8_t)(s))

#define SDP_DATA_ELEMENT_TYPE_UNSIGNED_INTEGER      0b00001000
#define SDP_DATA_ELEMENT_TYPE_SIGNED_INTEGER        0b00010000
#define SDP_DATA_ELEMENT_TYPE_UUID                  0b00011000
#define SDP_DATA_ELEMENT_TYPE_TEXT                  0b00100000
#define SDP_DATA_ELEMENT_TYPE_BOOLEAN               0b00101000
#define SDP_DATA_ELEMENT_TYPE_ELEMENT_SEQUENCE      0b00110000
#define SDP_DATA_ELEMENT_TYPE_ELEMENT_ALTERNATIVE   0b00111000
#define SDP_DATA_ELEMENT_TYPE_URL                   0b01000000
#define SDP_DATA_ELEMENT_TYPE(a)             ((a) & 0b11111000)

#define SDP_DATA_ELEMENT_SIZE_8BIT                  0b00000000
#define SDP_DATA_ELEMENT_SIZE_16BIT                 0b00000001
#define SDP_DATA_ELEMENT_SIZE_32BIT                 0b00000010
#define SDP_DATA_ELEMENT_SIZE_64BIT                 0b00000011
#define SDP_DATA_ELEMENT_SIZE_128BIT                0b00000100
#define SDP_DATA_ELEMENT_SIZE_VARIABLE_8            0b00000101
#define SDP_DATA_ELEMENT_SIZE_VARIABLE_16           0b00000110
#define SDP_DATA_ELEMENT_SIZE_VARIABLE_32           0b00000111
#define SDP_DATA_ELEMENT_SIZE(a)             ((a) & 0b00000111)

#define SDP_DATA_UINT8          (SDP_DATA_ELEMENT_TYPE_UNSIGNED_INTEGER | SDP_DATA_ELEMENT_SIZE_8BIT)
#define SDP_DATA_UINT16         (SDP_DATA_ELEMENT_TYPE_UNSIGNED_INTEGER | SDP_DATA_ELEMENT_SIZE_16BIT)
#define SDP_DATA_UINT32         (SDP_DATA_ELEMENT_TYPE_UNSIGNED_INTEGER | SDP_DATA_ELEMENT_SIZE_32BIT)
#define SDP_DATA_UUID16         (SDP_DATA_ELEMENT_TYPE_UUID             | SDP_DATA_ELEMENT_SIZE_16BIT)
#define SDP_DATA_UUID32         (SDP_DATA_ELEMENT_TYPE_UUID             | SDP_DATA_ELEMENT_SIZE_32BIT)
#define SDP_DATA_STRING8        (SDP_DATA_ELEMENT_TYPE_TEXT             | SDP_DATA_ELEMENT_SIZE_VARIABLE_8)
#define SDP_DATA_BOOL8          (SDP_DATA_ELEMENT_TYPE_BOOLEAN          | SDP_DATA_ELEMENT_SIZE_8BIT)
#define SDP_DATA_ELEMENT8       (SDP_DATA_ELEMENT_TYPE_ELEMENT_SEQUENCE | SDP_DATA_ELEMENT_SIZE_VARIABLE_8)
#define SDP_DATA_ELEMENT16      (SDP_DATA_ELEMENT_TYPE_ELEMENT_SEQUENCE | SDP_DATA_ELEMENT_SIZE_VARIABLE_16)

#define SDP_DATA_NULL           0
#define SDP_DATA_UINT8_2(a)     SDP_DATA_UINT8,a
#define SDP_DATA_UINT16_3(a)    SDP_DATA_UINT16,(uint8_t)(a>>8),(uint8_t)a
#define SDP_DATA_UINT32_5(a)    SDP_DATA_UINT32,(uint8_t)(a>>24),(uint8_t)(a>>16),(uint8_t)(a>>8),(uint8_t)a
#define SDP_DATA_UUID16_3(a)    SDP_DATA_UUID16,(uint8_t)(a>>8),(uint8_t)a
#define SDP_DATA_UUID32_5(a)    SDP_DATA_UUID32,(uint8_t)(a>>24),(uint8_t)(a>>16),(uint8_t)(a>>8),(uint8_t)a
#define SDP_DATA_BOOL8_2_TRUE   SDP_DATA_BOOL8, 1
#define SDP_DATA_BOOL8_2_FALSE  SDP_DATA_BOOL8, 0
#define SDP_DATA_ELEMENT8_2(a)  SDP_DATA_ELEMENT8,a
#define SDP_DATA_ELEMENT16_3(a) SDP_DATA_ELEMENT16,(uint8_t)(a>>8),(uint8_t)a

//
//  https://www.bluetooth.org/Technical/AssignedNumbers/service_discovery.htm
//

//                                          UUID16
#define SDP_PROTOCOL_SDP                    0x0001  // Bluetooth core spec
#define SDP_PROTOCOL_UDP                    0x0002  //
#define SDP_PROTOCOL_RFCOMM                 0x0003  // RFCOMM with TS 07.10
#define SDP_PROTOCOL_TCP                    0x0004  //
#define SDP_PROTOCOL_TCS_BIN                0x0005  // Telephony Control spec / TCS binary[DEPRECATED]
#define SDP_PROTOCOL_TCS_AT                 0x0006  //
#define SDP_PROTOCOL_ATT                    0x0007  // Attribute Protocol
#define SDP_PROTOCOL_OBEX                   0x0008  // IrDA Interoperability
#define SDP_PROTOCOL_IP                     0x0009  //
#define SDP_PROTOCOL_FTP                    0x000A  //
#define SDP_PROTOCOL_HTTP                   0x000C  //
#define SDP_PROTOCOL_WSP                    0x000E  //
#define SDP_PROTOCOL_BNEP                   0x000F  // Bluetooth Network Encapsulation Protocol (BNEP)
#define SDP_PROTOCOL_UPNP                   0x0010  // Extended Service Discovery Profile (ESDP)[DEPRECATED]
#define SDP_PROTOCOL_HIDP                   0x0011  // Human Interface Device Profile (HID)
#define SDP_PROTOCOL_HardcopyControlChannel 0x0012  // Hardcopy Cable Replacement Profile (HCRP)
#define SDP_PROTOCOL_HardcopyDataChannel    0x0014  // See Hardcopy Cable Replacement Profile (HCRP)
#define SDP_PROTOCOL_HardcopyNotification   0x0016  // Hardcopy Cable Replacement Profile (HCRP)
#define SDP_PROTOCOL_AVCTP                  0x0017  // Audio/Video Control Transport Protocol (AVCTP)
#define SDP_PROTOCOL_AVDTP                  0x0019  // Audio/Video Distribution Transport Protocol (AVDTP)
#define SDP_PROTOCOL_CMTP                   0x001B  // Common ISDN Access Profile (CIP)[DEPRECATED]
#define SDP_PROTOCOL_MCAPControlChannel     0x001E  // Multi-Channel Adaptation Protocol (MCAP)
#define SDP_PROTOCOL_MCAPDataChannel        0x001F  // Multi-Channel Adaptation Protocol (MCAP)
#define SDP_PROTOCOL_L2CAP                  0x0100  // Bluetooth core spec

//                                                              UUID16
#define SDP_SERVICE_CLASS_ServiceDiscoveryServerServiceClassID  0x1000  // Bluetooth Core Specification	Service Class
#define SDP_SERVICE_CLASS_BrowseGroupDescriptorServiceClassID   0x1001  // Bluetooth Core Specification	Service Class
#define SDP_SERVICE_CLASS_PublicBrowseRoot                      0x1002  // Bluetooth Core Specification	Service Class
#define SDP_SERVICE_CLASS_SerialPort                            0x1101  // Serial Port Profile (SPP)
#define SDP_SERVICE_CLASS_LANAccessUsingPPP                     0x1102  // LAN Access Profile[DEPRECATED]
#define SDP_SERVICE_CLASS_DialupNetworking                      0x1103  // Dial-up Networking Profile (DUN)
#define SDP_SERVICE_CLASS_IrMCSync                              0x1104  // Synchronization Profile (SYNC)
#define SDP_SERVICE_CLASS_OBEXObjectPush                        0x1105  // Object Push Profile (OPP)
#define SDP_SERVICE_CLASS_OBEXFileTransfer                      0x1106  // File Transfer Profile (FTP)
#define SDP_SERVICE_CLASS_IrMCSyncCommand                       0x1107  // Synchronization Profile (SYNC)
#define SDP_SERVICE_CLASS_Headset                               0x1108  // Headset Profile (HSP)
#define SDP_SERVICE_CLASS_CordlessTelephony                     0x1109  // Cordless Telephony Profile (CTP)
#define SDP_SERVICE_CLASS_AudioSource                           0x110A  // Advanced Audio Distribution Profile (A2DP)	Service Class
#define SDP_SERVICE_CLASS_AudioSink                             0x110B  // Advanced Audio Distribution Profile (A2DP)	Service Class
#define SDP_SERVICE_CLASS_A_V_RemoteControlTarget               0x110C  // Audio/Video Remote Control Profile (AVRCP)	Service Class
#define SDP_SERVICE_CLASS_AdvancedAudioDistribution             0x110D  // Advanced Audio Distribution Profile (A2DP)	Profile
#define SDP_SERVICE_CLASS_A_V_RemoteControl                     0x110E  // Audio/Video Remote Control Profile (AVRCP)
#define SDP_SERVICE_CLASS_A_V_RemoteControlController           0x110F  // Audio/Video Remote Control Profile (AVRCP)
#define SDP_SERVICE_CLASS_Intercom                              0x1110  // Intercom Profile (ICP)
#define SDP_SERVICE_CLASS_Fax                                   0x1111  // Fax Profile (FAX)
#define SDP_SERVICE_CLASS_Headset_Audio_Gateway_AG              0x1112  // Headset Profile (HSP)	Service Class
#define SDP_SERVICE_CLASS_WAP                                   0x1113  // Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_WAP_CLIENT                            0x1114  // Interoperability Requirements for Bluetooth technology as a WAP, Bluetooth SIG [DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_PANU                                  0x1115  // Personal Area Networking Profile (PAN)
#define SDP_SERVICE_CLASS_NAP                                   0x1116  // Personal Area Networking Profile (PAN)
#define SDP_SERVICE_CLASS_GN                                    0x1117  // Personal Area Networking Profile (PAN)
#define SDP_SERVICE_CLASS_DirectPrinting                        0x1118  // Basic Printing Profile (BPP)	Service Class
#define SDP_SERVICE_CLASS_ReferencePrinting                     0x1119  // See Basic Printing Profile (BPP)	Service Class
#define SDP_SERVICE_CLASS_Basic_Imaging_Profile                 0x111A  // Basic Imaging Profile (BIP)	Profile
#define SDP_SERVICE_CLASS_ImagingResponder                      0x111B  // Basic Imaging Profile (BIP)	Service Class
#define SDP_SERVICE_CLASS_ImagingAutomaticArchive               0x111C  // Basic Imaging Profile (BIP)	Service Class
#define SDP_SERVICE_CLASS_ImagingReferencedObjects              0x111D  // Basic Imaging Profile (BIP)	Service Class
#define SDP_SERVICE_CLASS_Handsfree                             0x111E  // Hands-Free Profile (HFP)
#define SDP_SERVICE_CLASS_HandsfreeAudioGateway                 0x111F  // Hands-free Profile (HFP)	Service Class
#define SDP_SERVICE_CLASS_DirectPrintingReferenceObjectsService 0x1120  // Basic Printing Profile (BPP)	Service Class
#define SDP_SERVICE_CLASS_ReflectedUI                           0x1121  // Basic Printing Profile (BPP)	Service Class
#define SDP_SERVICE_CLASS_BasicPrinting                         0x1122  // Basic Printing Profile (BPP)	Profile
#define SDP_SERVICE_CLASS_PrintingStatus                        0x1123  // Basic Printing Profile (BPP)	Service Class
#define SDP_SERVICE_CLASS_HumanInterfaceDeviceService           0x1124  // Human Interface Device (HID)
#define SDP_SERVICE_CLASS_HardcopyCableReplacement              0x1125  // Hardcopy Cable Replacement Profile (HCRP)	Profile
#define SDP_SERVICE_CLASS_HCR_Print                             0x1126  // Hardcopy Cable Replacement Profile (HCRP)	Service Class
#define SDP_SERVICE_CLASS_HCR_Scan                              0x1127  // Hardcopy Cable Replacement Profile (HCRP)	Service Class
#define SDP_SERVICE_CLASS_Common_ISDN_Access                    0x1128  // Common ISDN Access Profile (CIP)
#define SDP_SERVICE_CLASS_SIM_Access                            0x112D  // SIM Access Profile (SAP)
#define SDP_SERVICE_CLASS_Phonebook_Access_PCE                  0x112E  // Phonebook Access Profile (PBAP)	Service Class
#define SDP_SERVICE_CLASS_Phonebook_Access_PSE                  0x112F  // Phonebook Access Profile (PBAP)	Service Class
#define SDP_SERVICE_CLASS_Phonebook_Access                      0x1130  // Phonebook Access Profile (PBAP)	Profile
#define SDP_SERVICE_CLASS_Headset_HS                            0x1131  // Headset Profile (HSP)
#define SDP_SERVICE_CLASS_Message_Access_Server                 0x1132  // Message Access Profile (MAP)	Service Class
#define SDP_SERVICE_CLASS_Message_Notification_Server           0x1133  // Message Access Profile (MAP)	Service Class
#define SDP_SERVICE_CLASS_Message_Access_Profile                0x1134  // Message Access Profile (MAP)	Profile
#define SDP_SERVICE_CLASS_GNSS                                  0x1135  // Global Navigation Satellite System Profile (GNSS)	Profile
#define SDP_SERVICE_CLASS_GNSS_Server                           0x1136  // Global Navigation Satellite System Profile (GNSS)	Service Class
#define SDP_SERVICE_CLASS_3D_Display                            0x1137  // ?3D Synchronization Profile (3DSP)	Service Class
#define SDP_SERVICE_CLASS_3D_Glasses                            0x1138  // ?3D Synchronization Profile (3DSP)	?Service Class?#define SDP_SERVICE_CLASS_3D_Synchronization           0x1139?  // ?3D Synchronization Profile (3DSP)	?Profile
#define SDP_SERVICE_CLASS_MPS_Profile_UUID                      0x113A  // ?Multi-Profile Specification (MPS)	?Profile
#define SDP_SERVICE_CLASS_MPS_SC_UUID                           0x113B  // ?Multi-Profile Specification (MPS)	?Service Class
#define SDP_SERVICE_CLASS_CTN_Access_Service                    0x113C  // ?Calendar, Task, and Notes (CTN) Profile	?Service Class
#define SDP_SERVICE_CLASS_CTN_Notification_Service              0x113D  // ??Calendar Tasks and Notes (CTN) Profile	?Service Class
#define SDP_SERVICE_CLASS_CTN_Profile                           0x113E	// Calendar Tasks and Notes (CTN) Profile	?Profile
#define SDP_SERVICE_CLASS_PnPInformation                        0x1200  // Device Identification (DID)
#define SDP_SERVICE_CLASS_GenericNetworking                     0x1201  // N/A	Service Class
#define SDP_SERVICE_CLASS_GenericFileTransfer                   0x1202  // N/A	Service Class
#define SDP_SERVICE_CLASS_GenericAudio                          0x1203  // N/A	Service Class
#define SDP_SERVICE_CLASS_GenericTelephony                      0x1204  // N/A	Service Class
#define SDP_SERVICE_CLASS_UPNP_Service                          0x1205  // Enhanced Service Discovery Profile (ESDP) [DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_UPNP_IP_Service                       0x1206  // Enhanced Service Discovery Profile (ESDP) [DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_ESDP_UPNP_IP_PAN                      0x1300  // Enhanced Service Discovery Profile (ESDP) [DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_ESDP_UPNP_IP_LAP                      0x1301  // Enhanced Service Discovery Profile (ESDP)[DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_ESDP_UPNP_L2CAP                       0x1302  // Enhanced Service Discovery Profile (ESDP)[DEPRECATED]	Service Class
#define SDP_SERVICE_CLASS_VideoSource                           0x1303  // Video Distribution Profile (VDP)	Service Class
#define SDP_SERVICE_CLASS_VideoSink                             0x1304  // Video Distribution Profile (VDP)	Service Class
#define SDP_SERVICE_CLASS_VideoDistribution                     0x1305  // Video Distribution Profile (VDP)	Profile
#define SDP_SERVICE_CLASS_HDP                                   0x1400  // Health Device Profile	Profile
#define SDP_SERVICE_CLASS_HDP_Source                            0x1401  // Health Device Profile (HDP)	Service Class
#define SDP_SERVICE_CLASS_HDP_Sink                              0x1402  // Health Device Profile (HDP)	Service Class

// Bluetooth Core Specification: Universal Attributes
#define SDP_ATTRIBUTE_ServiceRecordHandle               0x0000  // uint32
#define SDP_ATTRIBUTE_ServiceClassIDList                0x0001  // element seq (uuid x n)
#define SDP_ATTRIBUTE_ServiceRecordState                0x0002  // uint32
#define SDP_ATTRIBUTE_ServiceID                         0x0003  // uuid
#define SDP_ATTRIBUTE_ProtocolDescriptorList            0x0004  // elemnt seq or alter
#define SDP_ATTRIBUTE_BrowseGroupList                   0x0005  // elem seq (uuid x n)
#define SDP_ATTRIBUTE_LanguageBaseAttributeIDList       0x0006  // elem seq (uint16 x 3)
#define SDP_ATTRIBUTE_ServiceInfoTimeToLive             0x0007  // uint32
#define SDP_ATTRIBUTE_ServiceAvailability               0x0008  // uint8
#define SDP_ATTRIBUTE_BluetoothProfileDescriptorList    0x0009  // elem seq
#define SDP_ATTRIBUTE_DocumentationURL                  0x000a  // elem seq
#define SDP_ATTRIBUTE_ClientExecutableURL               0x000b  // elem seq
#define SDP_ATTRIBUTE_IconURL                           0x000c  // elem seq
#define SDP_ATTRIBUTE_AdditionalProtocolDescriptorList  0x000d  // element seq
#define SDP_ATTRIBUTE_ServiceName                       0x0100  // text
#define SDP_ATTRIBUTE_ServiceDescription                0x0101  // text

// ServiceDiscoveryServerServiceClassID
#define SDP_ATTRIBUTE_VersionNumberList                 0x0200
#define SDP_ATTRIBUTE_ServiceDatabaseState              0x0201

// BrowseGroupDescriptorServiceClassID
#define SDP_ATTRIBUTE_GroupID                           0x0200

// PnPInformation (DeviceID_SPEC_V13.pdf)
#define SDP_ATTRIBUTE_DID_SpecificationID               0x0200  // uint16
#define SDP_ATTRIBUTE_DID_VendorID                      0x0201  // uint16
#define SDP_ATTRIBUTE_DID_ProductID                     0x0202  // uint16
#define SDP_ATTRIBUTE_DID_Version                       0x0203  // uint16
#define SDP_ATTRIBUTE_DID_PrimaryRecord                 0x0204  // bool8
#define SDP_ATTRIBUTE_DID_VenderIDSource                0x0205  // uint16
#define SDP_DID_SpecificationID                         0x0103  // V1.3
#define SDP_DID_VenderIDSource_Bluetooth                0x0001
#define SDP_DID_VenderIDSource_USB                      0x0002

// ADD START 2017.03.10
#define SDP_ATTRIBUTE_ID_OFFSET_ServiceName             0x0000
#define SDP_ATTRIBUTE_ID_OFFSET_ServiceDescription      0x0001
#define SDP_ATTRIBUTE_ID_OFFSET_ProviderName            0x0002

#define SDP_ATTRIBUTE_GoepL2capPsm                      0x0200
#define SDP_ATTRIBUTE_HIDDeviceReleaseNumber            0x0200
#define SDP_ATTRIBUTE_MPSDScenarios                     0x0200
#define SDP_ATTRIBUTE_SupportFeaturesList               0x0200
#define SDP_ATTRIBUTE_SupportedFeature                  0x0200
#define SDP_ATTRIBUTE_HIDParserVersion                  0x0201
#define SDP_ATTRIBUTE_MPMDScenarios                     0x0201
#define SDP_ATTRIBUTE_HIDDeviceSubclass                 0x0202
#define SDP_ATTRIBUTE_SupportedProfilesProtocols        0x0202
#define SDP_ATTRIBUTE_HIDCountryCode                    0x0203
#define SDP_ATTRIBUTE_HIDVirtualCable                   0x0204
#define SDP_ATTRIBUTE_HIDReconnectInitiate              0x0205
#define SDP_ATTRIBUTE_HIDDescriptorList                 0x0206
#define SDP_ATTRIBUTE_HIDLANGIDBaseList                 0x0207
#define SDP_ATTRIBUTE_HIDSDPDisable                     0x0208
#define SDP_ATTRIBUTE_HIDBatteryPower                   0x0209
#define SDP_ATTRIBUTE_HIDRemoteWake                     0x020A
#define SDP_ATTRIBUTE_HIDProfileVersion                 0x020B
#define SDP_ATTRIBUTE_HIDSupervisionTimeout             0x020C
#define SDP_ATTRIBUTE_HIDNormallyConnectable            0x020D
#define SDP_ATTRIBUTE_HIDBootDevice                     0x020E
#define SDP_ATTRIBUTE_HIDSSRHostMaxLatency              0x020F
#define SDP_ATTRIBUTE_HIDSSRHostMinTimeout              0x0210
#define SDP_ATTRIBUTE_ReservedforHIDAttributes          0x0211-0x03FF

#define SDP_ATTRIBUTE_ServiceVersion                    0x0300
#define SDP_ATTRIBUTE_DataExchangeSpecification         0x0301
#define SDP_ATTRIBUTE_ExternalNetwork                   0x0301
#define SDP_ATTRIBUTE_Network                           0x0301
#define SDP_ATTRIBUTE_SupportedDataStoresList           0x0301
#define SDP_ATTRIBUTE_DeviceName                        0x0302
#define SDP_ATTRIBUTE_FaxClass1Support                  0x0302
#define SDP_ATTRIBUTE_MCAPSupportedProcedures           0x0302
#define SDP_ATTRIBUTE_RemoteAudioVolumeControl          0x0302
#define SDP_ATTRIBUTE_FaxClass2_0Support                0x0303
#define SDP_ATTRIBUTE_SupportedFormatsList              0x0303
#define SDP_ATTRIBUTE_FaxClass2Support                  0x0304
#define SDP_ATTRIBUTE_FriendlyName                      0x0304
#define SDP_ATTRIBUTE_AudioFeedbackSupport              0x0305
#define SDP_ATTRIBUTE_DeviceLocation                    0x0306
#define SDP_ATTRIBUTE_NetworkAddress                    0x0306
#define SDP_ATTRIBUTE_WAPGateway                        0x0307
#define SDP_ATTRIBUTE_HomePageURL                       0x0308
#define SDP_ATTRIBUTE_WAPStackType                      0x0309
#define SDP_ATTRIBUTE_SecurityDescription               0x030A
#define SDP_ATTRIBUTE_NetAccessType                     0x030B
#define SDP_ATTRIBUTE_MaxNetAccessrate                  0x030C
#define SDP_ATTRIBUTE_IPv4Subnet                        0x030D
#define SDP_ATTRIBUTE_IPv6Subnet                        0x030E
#define SDP_ATTRIBUTE_SupportedCapabilities             0x0310
#define SDP_ATTRIBUTE_SupportedFeatures                 0x0311
#define SDP_ATTRIBUTE_SupportedFunctions                0x0312
#define SDP_ATTRIBUTE_TotalImagingDataCapacity          0x0313
#define SDP_ATTRIBUTE_SupportedRepositories             0x0314
#define SDP_ATTRIBUTE_CASInstanceID                     0x0315
#define SDP_ATTRIBUTE_MASInstanceID                     0x0315
#define SDP_ATTRIBUTE_SupportedMessageTypes             0x0316
#define SDP_ATTRIBUTE_CTNSupportedFeatures              0x0317
#define SDP_ATTRIBUTE_MapSupportedFeatures              0x0317
#define SDP_ATTRIBUTE_DocumentFormatsSupported          0x0350
#define SDP_ATTRIBUTE_CharacterRepertoiresSupported     0x0352
#define SDP_ATTRIBUTE_XHTMLPrintImageFormatsSupported   0x0354
#define SDP_ATTRIBUTE_ColorSupported                    0x0356
#define SDP_ATTRIBUTE_1284ID                            0x0358
#define SDP_ATTRIBUTE_PrinterName                       0x035A
#define SDP_ATTRIBUTE_PrinterLocation                   0x035C
#define SDP_ATTRIBUTE_DuplexSupported                   0x035E
#define SDP_ATTRIBUTE_MediaTypesSupported               0x0360
#define SDP_ATTRIBUTE_MaxMediaWidth                     0x0362
#define SDP_ATTRIBUTE_MaxMediaLength                    0x0364
#define SDP_ATTRIBUTE_EnhancedLayoutSupported           0x0366
#define SDP_ATTRIBUTE_RUIFormatsSupported               0x0368
#define SDP_ATTRIBUTE_ReferencePrintingRUISupported     0x0370
#define SDP_ATTRIBUTE_DirectPrintingRUISupported        0x0372
#define SDP_ATTRIBUTE_ReferencePrintingTopURL           0x0374
#define SDP_ATTRIBUTE_DirectPrintingTopURL              0x0376
#define SDP_ATTRIBUTE_PrinterAdminRUITopURL             0x0378
#define SDP_ATTRIBUTE_DeviceName_BPP                    0x037A
#define SDP_ATTRIBUTE_AvailableforHIDLanguageStrings    0x0400-0xFFFF

// ADD END 2017.03.10

#endif	/* BT_SPEC_SDP_H */

